# encoding: utf-8
# frozen_string_literal: true

require 'open-uri'

LLVM_SYNTAX_URI = "https://raw.githubusercontent.com/llvm/llvm-project/master/llvm/lib/AsmParser/LLLexer.cpp"
LLVM_KEYWORDS_FILE = "./lib/rouge/lexers/llvm/keywords.rb"

namespace :builtins do
  task :llvm do
    input     = URI.open(LLVM_SYNTAX_URI) { |f| f.read }
    generator = Rouge::Tasks::Builtins::LLVM.new
    keywords  = generator.extract_keywords(input)
    output    = generator.render_output(keywords)

    File.write(LLVM_KEYWORDS_FILE, output)
  end
end

module Rouge
  module Tasks
    module Builtins
      class LLVM
        def extract_keywords(input)
          keywords = Hash.new { |h,k| h[k] = Array.new }
          kind = nil

          input.each_line(";") do |line|
            if line =~ /#define (.*?)\(/
              case $1
              when "KEYWORD"
                kind = "keywords"
              when "TYPEKEYWORD"
                kind = "types"
              when "INSTKEYWORD"
                kind = "instructions"
              else
                kind = nil
              end

              next
            end

            next unless kind && line =~ /KEYWORD\("?([^)",]+)/

            keywords[kind].push $1
          end

          keywords.transform_values! { |v| v.sort }
        end

        def render_output(keywords, &b)
          return enum_for(:render_output, keywords).to_a.join("\n") unless b

          yield "# encoding: utf-8"
          yield "# frozen_string_literal: true"
          yield ""
          yield "# DO NOT EDIT"
          yield "# This file is automatically generated by `rake builtins:llvm`."
          yield "# See tasks/builtins/llvm.rake for more info."
          yield ""
          yield "module Rouge"
          yield "  module Lexers"
          yield "    class LLVM"
          keywords.each do |k,v|
            yield "      def self.#{k}"
            yield "        @#{k} ||= Set.new #{v.inspect}"
            yield "      end"
            yield ""
          end
          yield "    end"
          yield "  end"
          yield "end"
        end
      end
    end
  end
end
