///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// solver wrapper: direct or iterative
//
#include "rheolef/cg.h"
#include "rheolef/gmres.h"
#include "rheolef/vec_expr_v2.h"

#include "solver_wrapper.h"
#include <boost/numeric/ublas/matrix.hpp>

namespace rheolef {
using namespace std;

template<class T, class M>
solver_wrapper_rep<T,M>::solver_wrapper_rep (const csr<T,M>& a, const solver_option& opt)
 : solver_abstract_rep<T,M>(opt),
  _is_direct((opt.iterative == 0) ||
             (opt.iterative == solver_option::decide && a.pattern_dimension() <= 2)),
  _is_sym    (a.is_symmetric()),
  _sa(),
  _a()
{
#if defined(_RHEOLEF_HAVE_FLOAT128)
  // only ldlt available yet, for sym matrix
  if (!_is_sym) { _is_direct = false; }
#elif !defined(_RHEOLEF_HAVE_TRILINOS) && !defined(_RHEOLEF_HAVE_PASTIX)
  // no builtin ilu0 available yet:
  if (!_is_sym) { _is_direct = true; }
#endif
  if (_is_direct) {
    if (_is_sym) { _sa = ldlt (a,opt); }
    else         { _sa = lu   (a,opt); }
  } else {
    _sa = eye_basic<T,M>();
    _a = a;
  }
}
template<class T, class M>
void
solver_wrapper_rep<T,M>::update_values (const csr<T,M>& a) 
{
  _sa.update_values (a);
  if (!_is_direct) { _a = a; }
}
template<class T, class M>
vec<T,M>
solver_wrapper_rep<T,M>::solve (const vec<T,M>& b) const
{
  if (_is_direct) { return _sa.solve (b); }
#ifdef TO_CLEAN
  size_type max_iter = option().max_iter;
  T tol              = option().tol;
  T tol_reached      = tol;
#endif // TO_CLEAN
  vec<T,M> x (b.ownership(), 0);
  if (_is_sym) {
    int status = cg (_a, x, b, _sa, option());
  } else {
    size_type m = option().krylov_dimension;
    boost::numeric::ublas::matrix<T> h(m+1,m+1);
    vec<T,sequential> dummy(m,0.0);
    int status = gmres (_a, x, b, _sa, h, dummy, option());
  }
  check_macro (option().residue <= sqrt(option().tol),
	"solver: precision "<<option().tol<<" not reached: get "<<option().residue
	<< " after " << option().max_iter << " iterations");
  if (option().residue > option().tol) warning_macro (
	"solver: precision "<<option().tol<<" not reached: get "<<option().residue
	<< " after " << option().max_iter << " iterations");
  return x;
}
template<class T, class M>
vec<T,M>
solver_wrapper_rep<T,M>::trans_solve (const vec<T,M>& b) const
{
  if (_is_direct) { return _sa.trans_solve (b); }

  error_macro ("iterative trans_solve: not yet");
  // TODO: wrap a as a*x return a.trans_mult(x) and the same with _sa.solve(b)
  return vec<T,M>();
}
template <class T, class M>
typename solver_wrapper_rep<T,M>::determinant_type
solver_wrapper_rep<T,M>::det() const
{
  check_macro (_is_direct, "undefined determinant computation for iterative solver (HINT: use direct method)");
  return _sa.det();
}
template<class T, class M>
solver_wrapper_rep<T,M>::~solver_wrapper_rep()
{
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

template class solver_wrapper_rep<Float,sequential>;
#ifdef _RHEOLEF_HAVE_MPI
template class solver_wrapper_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
