///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef.h"
using namespace rheolef;
using namespace std;

// --------------------------------------------------------------------
// scalar test
// --------------------------------------------------------------------
Float f1 (const point& x) {return x[0]; }
Float f2 (const point& x) {return sqr(x[0]); }

Float my_sqr_fct (const Float& x) { return x*x; }

struct my_sqr_class : unary_function<Float,Float> {
  my_sqr_class() {}
  Float operator() (const Float& x) const { return x*x; }
};
// --------------------------------------------------------------------
// vector test
// --------------------------------------------------------------------
point vf1 (const point& x) {return point(x[0]); }

Float my_norm2_fct (const point& v) { return dot(v,v); }

struct my_norm2_class : unary_function<point,Float> {
  my_norm2_class() {}
  Float operator() (const point& v) const { return dot(v,v); }
};
// --------------------------------------------------------------------
int main(int argc, char**argv) {
warning_macro ("coucou 1");
    geo omega(argv[1]);
    space Xh (omega, "P0");
    space V1h (omega, "P1");
    space V2h (omega, "P2");
    cout << setprecision(15);

    // 1rst order polynom f1
    //  Xh = P0 ==> gauss with 1 point in riesz_representer
    // => exact
    field f11h = interpolate(V1h, f1);
    field mf110h = riesz_representer (Xh, f11h);
    Float int_f110h = dot(mf110h, field(Xh,1.0));
    cout << "int1_f110h = " << int_f110h << endl;

    // 2rst order polynom f2 interpolated P1
    //  Xh = P0 ==> gauss with 1 point in riesz_representer
    // => inexact
    field f12h = interpolate(V1h, f2);
    field mf120h = riesz_representer (Xh, f12h);
    Float int_f120h = dot(mf120h, field(Xh,1.0));
    cout << "int_f120h = " << int_f120h << endl;

    // 2rst order polynom f2 interpolated P2
    //  Xh = P0 ==> gauss with 1 point in riesz_representer
    // => inexact
    field f22h = interpolate(V2h, f2);
    field mf220h = riesz_representer (Xh, f22h);
    Float int_f220h = dot(mf220h, field(Xh,1.0));
    cout << "int_f220h = " << int_f220h << endl;

    // 2rst order polynom f2 interpolated P2
    //  Xh = P0 and gauss with 2 point in riesz_representer
    // => exact
    quadrature_option qopt;
    qopt.set_family(quadrature_option::gauss);
    qopt.set_order(2);
    field mf222h = riesz_representer (Xh, f22h, qopt);
    Float int_f222h = dot(mf222h, field(Xh,1.0));
    cout << "int_f222h = " << int_f222h << endl;
    check_macro ((int_f222h - 1/Float(3)) < 1e-7, "unexpected");

    // 2rst order polynom sqr(f1) with sqr(f1) interpolated P1 by compose()
    //  Xh = P0 and gauss with 2 point in riesz_representer
    // => inexact
    field msqr_f112h = riesz_representer (Xh, compose(sqr,f11h), qopt);
    Float int_sqrf112h = dot(msqr_f112h, field(Xh,1.0));
    cout << "int_sqrf112h = " << int_sqrf112h << endl;

    // 2rst order polynom sqr(f1) with sqr(f1) exactly evaluated by exact_compose()
    //  Xh = P0 and gauss with 2 point in riesz_representer
    // => exact
    // NOTE: with a call to exact_compose() with a class-fct
    field msqr_f1_2h = riesz_representer (Xh, exact_compose(my_sqr_class(),f11h), qopt);
    Float int_sqrf1_2h = dot(msqr_f1_2h, field(Xh,1.0));
    cout << "int_sqrf1_2h = " << int_sqrf1_2h << endl;
    check_macro ((int_sqrf1_2h - 1/Float(3)) < 1e-7, "unexpected");

    // 2rst order polynom f2 exactly interpolated P2
    //  Xh = P0 and gauss with 2 point in riesz_representer
    // => exact
    field m_f2_2h = riesz_representer (Xh, f22h, qopt);
    Float int_f2_2h = dot(m_f2_2h, field(Xh,1.0));
    cout << "int_f2_2h = " << int_f2_2h << endl;
    check_macro ((int_f2_2h - 1/Float(3)) < 1e-7, "unexpected");

    // with weighted forms implementation: exact
    form m_w (V1h, Xh, "mass", f11h);
    field m_sqr_f1_2h_w = m_w*f11h;
    Float int_sqr_f1_2h_w = dot(m_sqr_f1_2h_w, field(Xh,1.0));
    cout << "int_sqr_f1_2h_w = " << int_sqr_f1_2h_w << endl;
    check_macro ((int_sqr_f1_2h_w - 1/Float(3)) < 1e-7, "unexpected");

    // inexact with sqrt(f1) : int_0^1 sqrt(x) dx = 2/3
    // NOTE: with a call to exact_compose() with a usual fct
    field msqrt_f1_2h = riesz_representer (Xh, exact_compose(::sqrt,f11h), qopt);
    Float int_sqrt_f1_2h = dot(msqrt_f1_2h, field(Xh,1.0));
    cout << "int_sqrt_f1_2h = " << int_sqrt_f1_2h << endl;

    // -------------------------------------------
    // vector valued fields and fcts
    // -------------------------------------------
    space VV1h (omega, "P1", "vector");
    field vf11h = interpolate(VV1h, vf1);
    field m_norm2_vf1_2h = riesz_representer (Xh, exact_compose(ptr_fun(my_norm2_fct),vector_field(vf11h)), qopt);
    Float int_norm2_vf1_2h = dot(m_norm2_vf1_2h, field(Xh,1.0));
    cout << "int_norm2_vf1_2h = " << int_norm2_vf1_2h << endl;
    check_macro ((int_norm2_vf1_2h - 1/Float(3)) < 1e-7, "unexpected");

#ifdef TODO
#endif // TODO
}
