// usage:
//   demo2 line.geo   | demo2_estimator 
//   demo2 square.geo | demo2_estimator 

#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

#undef USE_QUADRATURE

template <class Iterator>
Iterator
build_estimator (
	Iterator e,		// exact error
	Iterator last_e,
	Iterator k,		// estimated error
	Iterator t)		// quality of the estimator
{
    while (e != last_e) {
	Float val = 1;
        if (1 + (*e) != Float(1) && 1 + (*k) != Float(1)) {
	    val = fabs((*k)/(*e));
	    // val = min(val, 1/val);
        }
	e++;
	k++;
	*t++ = val;
    }
    return t;
}
size_t dim = 2;

Float
f (const point& x)
{
    return Float(dim)*sin(x[0]+x[1]+x[2]);
}
Float
u (const point& x)
{
    return sin(x[0]+x[1]+x[2]);
}
Float du_dxi (const point& x) { return cos(x[0]+x[1]+x[2]); }

string get_approx_grad (const string &Pk)
{
    if (Pk == "P1") return "P0";
    if (Pk == "P2") return "P1d";
    cerr << "unexpected approximation " << Pk << endl;
    exit (1);
}
string get_approx_err (const string &Pk)
{
    if (Pk == "P1") return "P1d";
    if (Pk == "P2") return "P2d";
    cerr << "unexpected approximation " << Pk << endl;
    exit (1);
}
enum action_type { 
	none, 
	grad, 			//  d_dx(uh)
	grad_proj, 		//  P^T*P(d_dx(uh))
	grad_exact, 		//  pi_h(du_dx)
	error, 			//  |pi_h(u) - uh|
	error_grad, 		//  |pi_h(du_dx) - d_dx(uh) |
	error_grad_proj, 	//  |pi_h(du_dx)) - P^T*P(d_dx(uh)) |
	residual_grad_proj, 	//  |d_dx(uh) - P^T*P(d_dx(uh)) |
	estimator 		//  (pi_h(du_dx)) - P^T*P(d_dx(uh))) / (pi_h(du_dx) - d_dx(uh))
	};
enum output_type { 
	output_norm, 
	output_field,
	output_abs_field 
	};

void
usage(const char* prog)
{
    cerr << "usage: " << get_basename(prog) << " "
         << "[-grad|-grad-proj|-grad-exact|-error|-error-grad|-error-grad-var|-error-grad-proj|-residual-grad-proj|-estimator] " 
         << "[-output-norm|-output-field] " 
         << endl;
    exit(1);
}
int main(int argc, char**argv)
{
    action_type action = residual_grad_proj;
    output_type out    = output_abs_field;
    for (int i = 1; i < argc; i++) {
	     if (strcmp (argv[i], "-grad") == 0)               { action = grad; }
	else if (strcmp (argv[i], "-grad-proj") == 0)          { action = grad_proj; }
	else if (strcmp (argv[i], "-grad-exact") == 0)         { action = grad_exact; }
	else if (strcmp (argv[i], "-error") == 0)              { action = error; }
	else if (strcmp (argv[i], "-error-grad") == 0)         { action = error_grad; }
	else if (strcmp (argv[i], "-error-grad-var") == 0)     { action = error_grad; }
	else if (strcmp (argv[i], "-error-grad-proj") == 0)    { action = error_grad_proj; }
	else if (strcmp (argv[i], "-residual-grad-proj") == 0) { action = residual_grad_proj; }
	else if (strcmp (argv[i], "-estimator") == 0)          { action = estimator; }
	else if (strcmp (argv[i], "-output-norm") == 0)        { out = output_norm; }
	else if (strcmp (argv[i], "-output-field") == 0)       { out = output_field; }
	else if (strcmp (argv[i], "-output-abs-field") == 0)   { out = output_abs_field; }
	else usage(argv[0]);
    }
    int digits10 = numeric_limits<Float>::digits10;
    cout << setprecision(digits10);

    // the approximate solution
    field uh;
    cin >> uh;

    // get space Vh where uh belives and mesh
    space Vh      = uh.get_space();
    geo   omega_h = Vh.get_geo();
    dim = omega_h.dimension();
    string Pk = Vh.get_approx();

    // the solution interpolated
    string approx_grad = get_approx_grad(Pk);
    field pi_h_u = interpolate(Vh,u);
    
    // the space of gradients
    space Th (omega_h, approx_grad);

    // the space of errors
    string approx_err = get_approx_err(Pk);
    space Eh (omega_h, approx_err);

    // build b(.,.) operators
    form b(Vh, Th, "d_dx0");

    // build mass and its invert on Th 
    form mt     (Th, Th, "mass");
    form inv_mt (Th, Th, "inv_mass");

    // compute the gradient field of uh
    field duh_dx0 = inv_mt*(b*uh);
    
    // zh = project P(k-1) discontinuous gradient in Pk continuous
    form proj (Th, Vh, "mass");
    field gp0h (Vh);
#ifndef 	USE_QUADRATURE
    form mv (Vh, Vh, "mass");
    ssk<Float> fact_mv = ldlt(mv.uu);
    gp0h.u = fact_mv.solve(proj.uu*duh_dx0.u);
#else // 	USE_QUADRATURE
    form_diag mv (Vh, "mass");
    gp0h.u = (1./mv.uu)*(proj.uu*duh_dx0.u);
#endif // 	USE_QUADRATURE

    // gh = project P(k-1)-discontinuous gradient in Pk-discontinuous
    form me     (Eh, Eh, "mass");
    form inv_me (Eh, Eh, "inv_mass");
    form pr0    (Th, Eh, "mass");
    field gh = inv_me*pr0*duh_dx0;
    
    // rh = project Pk-continuous gradient in Pk-discontinuous
    form pr1    (Vh, Eh, "mass");
    field gph = inv_me*pr1*gp0h;

    // exact error on the gradient (P0):
    field pi_h_du_dx0 = interpolate(Eh, du_dxi);
    field eh  = gh  - pi_h_du_dx0;
    field eph = gph - pi_h_du_dx0;
    field rh  = gph - gh;

    if (action == grad) {

        if (out == output_field || out == output_abs_field) {
	    cout << rheo << gh;
	}
	return 0;

    } else if (action == grad_proj) {

        if (out == output_field || out == output_abs_field) {
	    cout << rheo << gph;
	}
	return 0;

    } else if (action == grad_exact) {

        if (out == output_field || out == output_abs_field) {
	    cout << rheo << pi_h_du_dx0;
	}
	return 0;

    } else if (action == estimator) {

        // quality of the estimator
        field quality (Eh);
        build_estimator(eh.u.begin(),eh.u.end(),rh.u.begin(),quality.u.begin());

        cerr << "quality = " << max(quality.min(), 1/quality.max()) << endl;
        if (out == output_field || out == output_abs_field) {
            cout << rheo << quality;
	}
	return 0;

    } else if (action == error) {

        // exact error on the solution (P1) projected P0:
        field e0h = pi_h_u - uh;
	cerr << "error_inf "  << e0h.max_abs()        << endl;
	cerr << "error_l2  "  << sqrt(mv(e0h,e0h)) << endl;
	return 0;

    } else if (action == error_grad) {

	cerr << "error_inf "  << eh.max_abs()        << endl;
	cerr << "error_l2  "  << sqrt(me(eh,eh)) << endl;
        if (out == output_field) {
	    cout << rheo << eh;
        } else if (out == output_abs_field) {
	    cout << rheo << abs(eh);
	}
	return 0;

    } else if (action == error_grad_proj) {

        // exact error on the projected gradient
	cerr << "error_inf "  << eph.max_abs()        << endl;
	cerr << "error_l2  "  << sqrt(me(eph,eph)) << endl;
        if (out == output_field) {
	    cout << rheo << eph;
        } else if (out == output_abs_field) {
	    cout << rheo << abs(eph);
	}
	return 0;

    } else if (action == residual_grad_proj) {
	//  |d_dx(uh) - P^T*P(d_dx(uh)) |
	cerr << "error_inf "  << rh.max_abs()        << endl;
	cerr << "error_l2  "  << sqrt(me(rh,rh)) << endl;
        if (out == output_field) {
	    cout << rheo << rh;
        } else if (out == output_abs_field) {
	    cout << rheo << abs(rh);
	}
	return 0;
    }
    return 0;
}
