#| -*-Scheme-*-

Copyright (C) 1986, 1987, 1988, 1989, 1990, 1991, 1992, 1993, 1994,
    1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005,
    2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016,
    2017, 2018, 2019 Massachusetts Institute of Technology

This file is part of MIT/GNU Scheme.

MIT/GNU Scheme is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

MIT/GNU Scheme is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with MIT/GNU Scheme; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301,
USA.

|#

;;;; Tests of string operations

(declare (usual-integrations))

;;;; Tests adapted from the Larceny R7RS test suite:

(define-test 'larceny-string
  (lambda ()

    (assert-string-ci< "A" "z")
    (assert-string-ci< "A" "z")
    (assert-string-ci< "a" "Z")
    (assert-string-ci< "a" "Z")
    (assert-string-ci<= "A" "z")
    (assert-string-ci<= "A" "z")
    (assert-string-ci<= "Z" "z")
    (assert-string-ci<= "Z" "z")
    (assert-string-ci<= "a" "Z")
    (assert-string-ci<= "a" "Z")
    (assert-string-ci<= "z" "Z")
    (assert-string-ci<= "z" "Z")
    (assert-string-ci= "z" "Z")
    (assert-string-ci!= "z" "a")
    (assert-string-ci> "Z" "a")
    (assert-string-ci> "Z" "a")
    (assert-string-ci> "z" "A")
    (assert-string-ci> "z" "A")
    (assert-string-ci>= "Z" "a")
    (assert-string-ci>= "Z" "a")
    (assert-string-ci>= "Z" "z")
    (assert-string-ci>= "Z" "z")
    (assert-string-ci>= "z" "A")
    (assert-string-ci>= "z" "A")
    (assert-string-ci>= "z" "Z")
    (assert-string-ci>= "z" "Z")

    (assert-string= (string-upcase "Hi") "HI")
    (assert-string= (string-upcase "HI") "HI")
    (assert-string= (string-downcase "Hi") "hi")
    (assert-string= (string-downcase "hi") "hi")
    (assert-string= (string-foldcase "Hi") "hi")
    (assert-string= (string-foldcase "HI") "hi")
    (assert-string= (string-foldcase "hi") "hi")
    (assert-string= (string-downcase "STRASSE")  "strasse")

    (assert-string= (string-upcase "Stra\xDF;e") "STRASSE")
    (assert-string= (string-downcase "Stra\xDF;e") "stra\xDF;e")
    (assert-string= (string-foldcase "Stra\xDF;e") "strasse")
    (assert-string= (string-downcase "\x3A3;") "\x3C3;")

    (assert-string= (string-upcase "\x39E;\x391;\x39F;\x3A3;")
		    "\x39E;\x391;\x39F;\x3A3;")
    ;; Would be "\x3BE;\x3B1;\x3BF;\x3C2;" with final sigma
    (assert-string= (string-downcase "\x39E;\x391;\x39F;\x3A3;")
		    "\x3BE;\x3B1;\x3BF;\x3C3;")
    ;; Would be "\x3BE;\x3B1;\x3BF;\x3C3;\x3C2;" with final sigma
    (assert-string= (string-downcase "\x39E;\x391;\x39F;\x3A3;\x3A3;")
		    "\x3BE;\x3B1;\x3BF;\x3C3;\x3C3;")
    ;; Would be "\x3BE;\x3B1;\x3BF;\x3C2; \x3C3;" with final sigma
    (assert-string= (string-downcase "\x39E;\x391;\x39F;\x3A3; \x3A3;")
		    "\x3BE;\x3B1;\x3BF;\x3C3; \x3C3;")
    (assert-string= (string-foldcase "\x39E;\x391;\x39F;\x3A3;")
		    "\x3BE;\x3B1;\x3BF;\x3C3;")
    (assert-string= (string-upcase "\x3BE;\x3B1;\x3BF;\x3C3;")
		    "\x39E;\x391;\x39F;\x3A3;")
    (assert-string= (string-upcase "\x3BE;\x3B1;\x3BF;\x3C2;")
		    "\x39E;\x391;\x39F;\x3A3;")

    (assert-string= (string-downcase "A\x3A3;'x") ; ' is a MidLetter
		    "a\x3C3;'x")

    (assert-string-ci= "Strasse" "Stra\xDF;e")
    (assert-string-ci= "STRASSE" "Stra\xDF;e")
    (assert-string-ci= "\x3BE;\x3B1;\x3BF;\x3C2;" "\x39E;\x391;\x39F;\x3A3;")
    (assert-string-ci= "\x3BE;\x3B1;\x3BF;\x3C3;" "\x39E;\x391;\x39F;\x3A3;")))

(define-test 'string-builder
  (lambda ()
    (let ((end (length latin-alphabet)))
      (do ((i 0 (fix:+ i 1)))
	  ((not (fix:< i end)))
	(let ((chars (list-head latin-alphabet i)))
	  (let ((result (build-string chars)))
	    (assert-true (legacy-string? result))
	    (assert-string= result (chars->string chars))))
	(let ((strings (make-test-strings i latin-alphabet #f)))
	  (let ((result (build-string strings)))
	    (assert-true (legacy-string? result))
	    (assert-string= result (string-append* strings))))
	(let ((strings (make-test-strings i latin-alphabet #t)))
	  (let ((result (build-string strings)))
	    (assert-true (legacy-string? result))
	    (assert-string= result (string-append* strings))))))
    (let ((end (length greek-alphabet)))
      (do ((i 0 (fix:+ i 1)))
	  ((not (fix:< i end)))
	(let ((chars (list-head greek-alphabet i)))
	  (assert-string= (build-string chars)
			  (chars->string chars)))
	(let ((strings (make-test-strings i greek-alphabet #f)))
	  (assert-string= (build-string strings)
			  (string-append* strings)))
	(let ((strings (make-test-strings i greek-alphabet #t)))
	  (assert-string= (build-string strings)
			  (string-append* strings)))))))

(define legacy-string?
  (make-primitive-procedure 'string? 1))

(define latin-alphabet
  '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m
    #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))

(define greek-alphabet
  '(#\x3B1 #\x3B2 #\x3B3 #\x3B4 #\x3B5
    #\x3B6 #\x3B7 #\x3B8 #\x3B9 #\x3BA
    #\x3BB #\x3BC #\x3BD #\x3BE #\x3BF
    #\x3C0 #\x3C1 #\x3C2 #\x3C3 #\x3C4
    #\x3C5 #\x3C6 #\x3C7 #\x3C8 #\x3C9))

(define (build-string objects)
  (let ((builder (string-builder)))
    (for-each builder objects)
    (builder)))

(define (chars->string chars)
  (let ((s (make-string (length chars))))
    (do ((chars chars (cdr chars))
	 (i 0 (fix:+ i 1)))
	((not (pair? chars)))
      (string-set! s i (car chars)))
    s))

(define (make-test-strings n alphabet reverse?)
  (let loop ((k 0) (strings '()))
    (if (fix:< k n)
	(loop (fix:+ k 1)
	      (cons (chars->string (list-head alphabet k))
		    strings))
	(if reverse?
	    strings
	    (reverse! strings)))))

(define (run-break-test procedure test-case)
  (receive (string breaks) (convert-break-test-case test-case)
    (with-test-properties
     (lambda ()
       (assert-equal (procedure string) breaks))
     'expression string)))

(define (convert-break-test-case test-case)
  (let ((builder (string-builder)))
    (let loop ((test-case test-case) (index 0) (breaks '()))
      (let ((breaks
	     (if (car test-case)
		 (cons index breaks)
		 breaks)))
	(if (pair? (cdr test-case))
	    (begin
	      (builder (cadr test-case))
	      (loop (cddr test-case) (fix:+ index 1) breaks))
	    (values (builder 'immutable)
		    (reverse! breaks)))))))

(define-test 'grapheme-cluster-breaks
  (map (lambda (test-case)
	 (lambda ()
	   (run-break-test grapheme-cluster-breaks test-case)))
       '((#t #\x0020 #t #\x0020 #t)
	 (#t #\x0020 #f #\x0308 #t #\x0020 #t)
	 (#t #\x0020 #t #\x000D #t)
	 (#t #\x0020 #f #\x0308 #t #\x000D #t)
	 (#t #\x0020 #t #\x000A #t)
	 (#t #\x0020 #f #\x0308 #t #\x000A #t)
	 (#t #\x0020 #t #\x0001 #t)
	 (#t #\x0020 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0020 #f #\x0300 #t)
	 (#t #\x0020 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0020 #t #\x0600 #t)
	 (#t #\x0020 #f #\x0308 #t #\x0600 #t)
	 (#t #\x0020 #f #\x0903 #t)
	 (#t #\x0020 #f #\x0308 #f #\x0903 #t)
	 (#t #\x0020 #t #\x1100 #t)
	 (#t #\x0020 #f #\x0308 #t #\x1100 #t)
	 (#t #\x0020 #t #\x1160 #t)
	 (#t #\x0020 #f #\x0308 #t #\x1160 #t)
	 (#t #\x0020 #t #\x11A8 #t)
	 (#t #\x0020 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x0020 #t #\xAC00 #t)
	 (#t #\x0020 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x0020 #t #\xAC01 #t)
	 (#t #\x0020 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x0020 #t #\x1F1E6 #t)
	 (#t #\x0020 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0020 #t #\x261D #t)
	 (#t #\x0020 #f #\x0308 #t #\x261D #t)
	 (#t #\x0020 #t #\x1F3FB #t)
	 (#t #\x0020 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0020 #f #\x200D #t)
	 (#t #\x0020 #f #\x0308 #f #\x200D #t)
	 (#t #\x0020 #t #\x2764 #t)
	 (#t #\x0020 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0020 #t #\x1F466 #t)
	 (#t #\x0020 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0020 #t #\x0378 #t)
	 (#t #\x0020 #f #\x0308 #t #\x0378 #t)
	 (#t #\x0020 #t #\xD800 #t)
	 (#t #\x0020 #f #\x0308 #t #\xD800 #t)
	 (#t #\x000D #t #\x0020 #t)
	 (#t #\x000D #t #\x0308 #t #\x0020 #t)
	 (#t #\x000D #t #\x000D #t)
	 (#t #\x000D #t #\x0308 #t #\x000D #t)
	 (#t #\x000D #f #\x000A #t)
	 (#t #\x000D #t #\x0308 #t #\x000A #t)
	 (#t #\x000D #t #\x0001 #t)
	 (#t #\x000D #t #\x0308 #t #\x0001 #t)
	 (#t #\x000D #t #\x0300 #t)
	 (#t #\x000D #t #\x0308 #f #\x0300 #t)
	 (#t #\x000D #t #\x0600 #t)
	 (#t #\x000D #t #\x0308 #t #\x0600 #t)
	 (#t #\x000D #t #\x0903 #t)
	 (#t #\x000D #t #\x0308 #f #\x0903 #t)
	 (#t #\x000D #t #\x1100 #t)
	 (#t #\x000D #t #\x0308 #t #\x1100 #t)
	 (#t #\x000D #t #\x1160 #t)
	 (#t #\x000D #t #\x0308 #t #\x1160 #t)
	 (#t #\x000D #t #\x11A8 #t)
	 (#t #\x000D #t #\x0308 #t #\x11A8 #t)
	 (#t #\x000D #t #\xAC00 #t)
	 (#t #\x000D #t #\x0308 #t #\xAC00 #t)
	 (#t #\x000D #t #\xAC01 #t)
	 (#t #\x000D #t #\x0308 #t #\xAC01 #t)
	 (#t #\x000D #t #\x1F1E6 #t)
	 (#t #\x000D #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x000D #t #\x261D #t)
	 (#t #\x000D #t #\x0308 #t #\x261D #t)
	 (#t #\x000D #t #\x1F3FB #t)
	 (#t #\x000D #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x000D #t #\x200D #t)
	 (#t #\x000D #t #\x0308 #f #\x200D #t)
	 (#t #\x000D #t #\x2764 #t)
	 (#t #\x000D #t #\x0308 #t #\x2764 #t)
	 (#t #\x000D #t #\x1F466 #t)
	 (#t #\x000D #t #\x0308 #t #\x1F466 #t)
	 (#t #\x000D #t #\x0378 #t)
	 (#t #\x000D #t #\x0308 #t #\x0378 #t)
	 (#t #\x000D #t #\xD800 #t)
	 (#t #\x000D #t #\x0308 #t #\xD800 #t)
	 (#t #\x000A #t #\x0020 #t)
	 (#t #\x000A #t #\x0308 #t #\x0020 #t)
	 (#t #\x000A #t #\x000D #t)
	 (#t #\x000A #t #\x0308 #t #\x000D #t)
	 (#t #\x000A #t #\x000A #t)
	 (#t #\x000A #t #\x0308 #t #\x000A #t)
	 (#t #\x000A #t #\x0001 #t)
	 (#t #\x000A #t #\x0308 #t #\x0001 #t)
	 (#t #\x000A #t #\x0300 #t)
	 (#t #\x000A #t #\x0308 #f #\x0300 #t)
	 (#t #\x000A #t #\x0600 #t)
	 (#t #\x000A #t #\x0308 #t #\x0600 #t)
	 (#t #\x000A #t #\x0903 #t)
	 (#t #\x000A #t #\x0308 #f #\x0903 #t)
	 (#t #\x000A #t #\x1100 #t)
	 (#t #\x000A #t #\x0308 #t #\x1100 #t)
	 (#t #\x000A #t #\x1160 #t)
	 (#t #\x000A #t #\x0308 #t #\x1160 #t)
	 (#t #\x000A #t #\x11A8 #t)
	 (#t #\x000A #t #\x0308 #t #\x11A8 #t)
	 (#t #\x000A #t #\xAC00 #t)
	 (#t #\x000A #t #\x0308 #t #\xAC00 #t)
	 (#t #\x000A #t #\xAC01 #t)
	 (#t #\x000A #t #\x0308 #t #\xAC01 #t)
	 (#t #\x000A #t #\x1F1E6 #t)
	 (#t #\x000A #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x000A #t #\x261D #t)
	 (#t #\x000A #t #\x0308 #t #\x261D #t)
	 (#t #\x000A #t #\x1F3FB #t)
	 (#t #\x000A #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x000A #t #\x200D #t)
	 (#t #\x000A #t #\x0308 #f #\x200D #t)
	 (#t #\x000A #t #\x2764 #t)
	 (#t #\x000A #t #\x0308 #t #\x2764 #t)
	 (#t #\x000A #t #\x1F466 #t)
	 (#t #\x000A #t #\x0308 #t #\x1F466 #t)
	 (#t #\x000A #t #\x0378 #t)
	 (#t #\x000A #t #\x0308 #t #\x0378 #t)
	 (#t #\x000A #t #\xD800 #t)
	 (#t #\x000A #t #\x0308 #t #\xD800 #t)
	 (#t #\x0001 #t #\x0020 #t)
	 (#t #\x0001 #t #\x0308 #t #\x0020 #t)
	 (#t #\x0001 #t #\x000D #t)
	 (#t #\x0001 #t #\x0308 #t #\x000D #t)
	 (#t #\x0001 #t #\x000A #t)
	 (#t #\x0001 #t #\x0308 #t #\x000A #t)
	 (#t #\x0001 #t #\x0001 #t)
	 (#t #\x0001 #t #\x0308 #t #\x0001 #t)
	 (#t #\x0001 #t #\x0300 #t)
	 (#t #\x0001 #t #\x0308 #f #\x0300 #t)
	 (#t #\x0001 #t #\x0600 #t)
	 (#t #\x0001 #t #\x0308 #t #\x0600 #t)
	 (#t #\x0001 #t #\x0903 #t)
	 (#t #\x0001 #t #\x0308 #f #\x0903 #t)
	 (#t #\x0001 #t #\x1100 #t)
	 (#t #\x0001 #t #\x0308 #t #\x1100 #t)
	 (#t #\x0001 #t #\x1160 #t)
	 (#t #\x0001 #t #\x0308 #t #\x1160 #t)
	 (#t #\x0001 #t #\x11A8 #t)
	 (#t #\x0001 #t #\x0308 #t #\x11A8 #t)
	 (#t #\x0001 #t #\xAC00 #t)
	 (#t #\x0001 #t #\x0308 #t #\xAC00 #t)
	 (#t #\x0001 #t #\xAC01 #t)
	 (#t #\x0001 #t #\x0308 #t #\xAC01 #t)
	 (#t #\x0001 #t #\x1F1E6 #t)
	 (#t #\x0001 #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0001 #t #\x261D #t)
	 (#t #\x0001 #t #\x0308 #t #\x261D #t)
	 (#t #\x0001 #t #\x1F3FB #t)
	 (#t #\x0001 #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x0001 #t #\x200D #t)
	 (#t #\x0001 #t #\x0308 #f #\x200D #t)
	 (#t #\x0001 #t #\x2764 #t)
	 (#t #\x0001 #t #\x0308 #t #\x2764 #t)
	 (#t #\x0001 #t #\x1F466 #t)
	 (#t #\x0001 #t #\x0308 #t #\x1F466 #t)
	 (#t #\x0001 #t #\x0378 #t)
	 (#t #\x0001 #t #\x0308 #t #\x0378 #t)
	 (#t #\x0001 #t #\xD800 #t)
	 (#t #\x0001 #t #\x0308 #t #\xD800 #t)
	 (#t #\x0300 #t #\x0020 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0020 #t)
	 (#t #\x0300 #t #\x000D #t)
	 (#t #\x0300 #f #\x0308 #t #\x000D #t)
	 (#t #\x0300 #t #\x000A #t)
	 (#t #\x0300 #f #\x0308 #t #\x000A #t)
	 (#t #\x0300 #t #\x0001 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0300 #f #\x0300 #t)
	 (#t #\x0300 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0300 #t #\x0600 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0600 #t)
	 (#t #\x0300 #f #\x0903 #t)
	 (#t #\x0300 #f #\x0308 #f #\x0903 #t)
	 (#t #\x0300 #t #\x1100 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1100 #t)
	 (#t #\x0300 #t #\x1160 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1160 #t)
	 (#t #\x0300 #t #\x11A8 #t)
	 (#t #\x0300 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x0300 #t #\xAC00 #t)
	 (#t #\x0300 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x0300 #t #\xAC01 #t)
	 (#t #\x0300 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x0300 #t #\x1F1E6 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0300 #t #\x261D #t)
	 (#t #\x0300 #f #\x0308 #t #\x261D #t)
	 (#t #\x0300 #t #\x1F3FB #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0300 #f #\x200D #t)
	 (#t #\x0300 #f #\x0308 #f #\x200D #t)
	 (#t #\x0300 #t #\x2764 #t)
	 (#t #\x0300 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0300 #t #\x1F466 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0300 #t #\x0378 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0378 #t)
	 (#t #\x0300 #t #\xD800 #t)
	 (#t #\x0300 #f #\x0308 #t #\xD800 #t)
	 (#t #\x0600 #f #\x0020 #t)
	 (#t #\x0600 #f #\x0308 #t #\x0020 #t)
	 (#t #\x0600 #t #\x000D #t)
	 (#t #\x0600 #f #\x0308 #t #\x000D #t)
	 (#t #\x0600 #t #\x000A #t)
	 (#t #\x0600 #f #\x0308 #t #\x000A #t)
	 (#t #\x0600 #t #\x0001 #t)
	 (#t #\x0600 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0600 #f #\x0300 #t)
	 (#t #\x0600 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0600 #f #\x0600 #t)
	 (#t #\x0600 #f #\x0308 #t #\x0600 #t)
	 (#t #\x0600 #f #\x0903 #t)
	 (#t #\x0600 #f #\x0308 #f #\x0903 #t)
	 (#t #\x0600 #f #\x1100 #t)
	 (#t #\x0600 #f #\x0308 #t #\x1100 #t)
	 (#t #\x0600 #f #\x1160 #t)
	 (#t #\x0600 #f #\x0308 #t #\x1160 #t)
	 (#t #\x0600 #f #\x11A8 #t)
	 (#t #\x0600 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x0600 #f #\xAC00 #t)
	 (#t #\x0600 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x0600 #f #\xAC01 #t)
	 (#t #\x0600 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x0600 #f #\x1F1E6 #t)
	 (#t #\x0600 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0600 #f #\x261D #t)
	 (#t #\x0600 #f #\x0308 #t #\x261D #t)
	 (#t #\x0600 #f #\x1F3FB #t)
	 (#t #\x0600 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0600 #f #\x200D #t)
	 (#t #\x0600 #f #\x0308 #f #\x200D #t)
	 (#t #\x0600 #f #\x2764 #t)
	 (#t #\x0600 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0600 #f #\x1F466 #t)
	 (#t #\x0600 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0600 #f #\x0378 #t)
	 (#t #\x0600 #f #\x0308 #t #\x0378 #t)
	 (#t #\x0600 #t #\xD800 #t)
	 (#t #\x0600 #f #\x0308 #t #\xD800 #t)
	 (#t #\x0903 #t #\x0020 #t)
	 (#t #\x0903 #f #\x0308 #t #\x0020 #t)
	 (#t #\x0903 #t #\x000D #t)
	 (#t #\x0903 #f #\x0308 #t #\x000D #t)
	 (#t #\x0903 #t #\x000A #t)
	 (#t #\x0903 #f #\x0308 #t #\x000A #t)
	 (#t #\x0903 #t #\x0001 #t)
	 (#t #\x0903 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0903 #f #\x0300 #t)
	 (#t #\x0903 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0903 #t #\x0600 #t)
	 (#t #\x0903 #f #\x0308 #t #\x0600 #t)
	 (#t #\x0903 #f #\x0903 #t)
	 (#t #\x0903 #f #\x0308 #f #\x0903 #t)
	 (#t #\x0903 #t #\x1100 #t)
	 (#t #\x0903 #f #\x0308 #t #\x1100 #t)
	 (#t #\x0903 #t #\x1160 #t)
	 (#t #\x0903 #f #\x0308 #t #\x1160 #t)
	 (#t #\x0903 #t #\x11A8 #t)
	 (#t #\x0903 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x0903 #t #\xAC00 #t)
	 (#t #\x0903 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x0903 #t #\xAC01 #t)
	 (#t #\x0903 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x0903 #t #\x1F1E6 #t)
	 (#t #\x0903 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0903 #t #\x261D #t)
	 (#t #\x0903 #f #\x0308 #t #\x261D #t)
	 (#t #\x0903 #t #\x1F3FB #t)
	 (#t #\x0903 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0903 #f #\x200D #t)
	 (#t #\x0903 #f #\x0308 #f #\x200D #t)
	 (#t #\x0903 #t #\x2764 #t)
	 (#t #\x0903 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0903 #t #\x1F466 #t)
	 (#t #\x0903 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0903 #t #\x0378 #t)
	 (#t #\x0903 #f #\x0308 #t #\x0378 #t)
	 (#t #\x0903 #t #\xD800 #t)
	 (#t #\x0903 #f #\x0308 #t #\xD800 #t)
	 (#t #\x1100 #t #\x0020 #t)
	 (#t #\x1100 #f #\x0308 #t #\x0020 #t)
	 (#t #\x1100 #t #\x000D #t)
	 (#t #\x1100 #f #\x0308 #t #\x000D #t)
	 (#t #\x1100 #t #\x000A #t)
	 (#t #\x1100 #f #\x0308 #t #\x000A #t)
	 (#t #\x1100 #t #\x0001 #t)
	 (#t #\x1100 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1100 #f #\x0300 #t)
	 (#t #\x1100 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1100 #t #\x0600 #t)
	 (#t #\x1100 #f #\x0308 #t #\x0600 #t)
	 (#t #\x1100 #f #\x0903 #t)
	 (#t #\x1100 #f #\x0308 #f #\x0903 #t)
	 (#t #\x1100 #f #\x1100 #t)
	 (#t #\x1100 #f #\x0308 #t #\x1100 #t)
	 (#t #\x1100 #f #\x1160 #t)
	 (#t #\x1100 #f #\x0308 #t #\x1160 #t)
	 (#t #\x1100 #t #\x11A8 #t)
	 (#t #\x1100 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x1100 #f #\xAC00 #t)
	 (#t #\x1100 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x1100 #f #\xAC01 #t)
	 (#t #\x1100 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x1100 #t #\x1F1E6 #t)
	 (#t #\x1100 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1100 #t #\x261D #t)
	 (#t #\x1100 #f #\x0308 #t #\x261D #t)
	 (#t #\x1100 #t #\x1F3FB #t)
	 (#t #\x1100 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1100 #f #\x200D #t)
	 (#t #\x1100 #f #\x0308 #f #\x200D #t)
	 (#t #\x1100 #t #\x2764 #t)
	 (#t #\x1100 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1100 #t #\x1F466 #t)
	 (#t #\x1100 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1100 #t #\x0378 #t)
	 (#t #\x1100 #f #\x0308 #t #\x0378 #t)
	 (#t #\x1100 #t #\xD800 #t)
	 (#t #\x1100 #f #\x0308 #t #\xD800 #t)
	 (#t #\x1160 #t #\x0020 #t)
	 (#t #\x1160 #f #\x0308 #t #\x0020 #t)
	 (#t #\x1160 #t #\x000D #t)
	 (#t #\x1160 #f #\x0308 #t #\x000D #t)
	 (#t #\x1160 #t #\x000A #t)
	 (#t #\x1160 #f #\x0308 #t #\x000A #t)
	 (#t #\x1160 #t #\x0001 #t)
	 (#t #\x1160 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1160 #f #\x0300 #t)
	 (#t #\x1160 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1160 #t #\x0600 #t)
	 (#t #\x1160 #f #\x0308 #t #\x0600 #t)
	 (#t #\x1160 #f #\x0903 #t)
	 (#t #\x1160 #f #\x0308 #f #\x0903 #t)
	 (#t #\x1160 #t #\x1100 #t)
	 (#t #\x1160 #f #\x0308 #t #\x1100 #t)
	 (#t #\x1160 #f #\x1160 #t)
	 (#t #\x1160 #f #\x0308 #t #\x1160 #t)
	 (#t #\x1160 #f #\x11A8 #t)
	 (#t #\x1160 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x1160 #t #\xAC00 #t)
	 (#t #\x1160 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x1160 #t #\xAC01 #t)
	 (#t #\x1160 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x1160 #t #\x1F1E6 #t)
	 (#t #\x1160 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1160 #t #\x261D #t)
	 (#t #\x1160 #f #\x0308 #t #\x261D #t)
	 (#t #\x1160 #t #\x1F3FB #t)
	 (#t #\x1160 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1160 #f #\x200D #t)
	 (#t #\x1160 #f #\x0308 #f #\x200D #t)
	 (#t #\x1160 #t #\x2764 #t)
	 (#t #\x1160 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1160 #t #\x1F466 #t)
	 (#t #\x1160 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1160 #t #\x0378 #t)
	 (#t #\x1160 #f #\x0308 #t #\x0378 #t)
	 (#t #\x1160 #t #\xD800 #t)
	 (#t #\x1160 #f #\x0308 #t #\xD800 #t)
	 (#t #\x11A8 #t #\x0020 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x0020 #t)
	 (#t #\x11A8 #t #\x000D #t)
	 (#t #\x11A8 #f #\x0308 #t #\x000D #t)
	 (#t #\x11A8 #t #\x000A #t)
	 (#t #\x11A8 #f #\x0308 #t #\x000A #t)
	 (#t #\x11A8 #t #\x0001 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x0001 #t)
	 (#t #\x11A8 #f #\x0300 #t)
	 (#t #\x11A8 #f #\x0308 #f #\x0300 #t)
	 (#t #\x11A8 #t #\x0600 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x0600 #t)
	 (#t #\x11A8 #f #\x0903 #t)
	 (#t #\x11A8 #f #\x0308 #f #\x0903 #t)
	 (#t #\x11A8 #t #\x1100 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x1100 #t)
	 (#t #\x11A8 #t #\x1160 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x1160 #t)
	 (#t #\x11A8 #f #\x11A8 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x11A8 #t #\xAC00 #t)
	 (#t #\x11A8 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x11A8 #t #\xAC01 #t)
	 (#t #\x11A8 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x11A8 #t #\x1F1E6 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x11A8 #t #\x261D #t)
	 (#t #\x11A8 #f #\x0308 #t #\x261D #t)
	 (#t #\x11A8 #t #\x1F3FB #t)
	 (#t #\x11A8 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x11A8 #f #\x200D #t)
	 (#t #\x11A8 #f #\x0308 #f #\x200D #t)
	 (#t #\x11A8 #t #\x2764 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x2764 #t)
	 (#t #\x11A8 #t #\x1F466 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x11A8 #t #\x0378 #t)
	 (#t #\x11A8 #f #\x0308 #t #\x0378 #t)
	 (#t #\x11A8 #t #\xD800 #t)
	 (#t #\x11A8 #f #\x0308 #t #\xD800 #t)
	 (#t #\xAC00 #t #\x0020 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x0020 #t)
	 (#t #\xAC00 #t #\x000D #t)
	 (#t #\xAC00 #f #\x0308 #t #\x000D #t)
	 (#t #\xAC00 #t #\x000A #t)
	 (#t #\xAC00 #f #\x0308 #t #\x000A #t)
	 (#t #\xAC00 #t #\x0001 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x0001 #t)
	 (#t #\xAC00 #f #\x0300 #t)
	 (#t #\xAC00 #f #\x0308 #f #\x0300 #t)
	 (#t #\xAC00 #t #\x0600 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x0600 #t)
	 (#t #\xAC00 #f #\x0903 #t)
	 (#t #\xAC00 #f #\x0308 #f #\x0903 #t)
	 (#t #\xAC00 #t #\x1100 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x1100 #t)
	 (#t #\xAC00 #f #\x1160 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x1160 #t)
	 (#t #\xAC00 #f #\x11A8 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x11A8 #t)
	 (#t #\xAC00 #t #\xAC00 #t)
	 (#t #\xAC00 #f #\x0308 #t #\xAC00 #t)
	 (#t #\xAC00 #t #\xAC01 #t)
	 (#t #\xAC00 #f #\x0308 #t #\xAC01 #t)
	 (#t #\xAC00 #t #\x1F1E6 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\xAC00 #t #\x261D #t)
	 (#t #\xAC00 #f #\x0308 #t #\x261D #t)
	 (#t #\xAC00 #t #\x1F3FB #t)
	 (#t #\xAC00 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\xAC00 #f #\x200D #t)
	 (#t #\xAC00 #f #\x0308 #f #\x200D #t)
	 (#t #\xAC00 #t #\x2764 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x2764 #t)
	 (#t #\xAC00 #t #\x1F466 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x1F466 #t)
	 (#t #\xAC00 #t #\x0378 #t)
	 (#t #\xAC00 #f #\x0308 #t #\x0378 #t)
	 (#t #\xAC00 #t #\xD800 #t)
	 (#t #\xAC00 #f #\x0308 #t #\xD800 #t)
	 (#t #\xAC01 #t #\x0020 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x0020 #t)
	 (#t #\xAC01 #t #\x000D #t)
	 (#t #\xAC01 #f #\x0308 #t #\x000D #t)
	 (#t #\xAC01 #t #\x000A #t)
	 (#t #\xAC01 #f #\x0308 #t #\x000A #t)
	 (#t #\xAC01 #t #\x0001 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x0001 #t)
	 (#t #\xAC01 #f #\x0300 #t)
	 (#t #\xAC01 #f #\x0308 #f #\x0300 #t)
	 (#t #\xAC01 #t #\x0600 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x0600 #t)
	 (#t #\xAC01 #f #\x0903 #t)
	 (#t #\xAC01 #f #\x0308 #f #\x0903 #t)
	 (#t #\xAC01 #t #\x1100 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x1100 #t)
	 (#t #\xAC01 #t #\x1160 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x1160 #t)
	 (#t #\xAC01 #f #\x11A8 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x11A8 #t)
	 (#t #\xAC01 #t #\xAC00 #t)
	 (#t #\xAC01 #f #\x0308 #t #\xAC00 #t)
	 (#t #\xAC01 #t #\xAC01 #t)
	 (#t #\xAC01 #f #\x0308 #t #\xAC01 #t)
	 (#t #\xAC01 #t #\x1F1E6 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\xAC01 #t #\x261D #t)
	 (#t #\xAC01 #f #\x0308 #t #\x261D #t)
	 (#t #\xAC01 #t #\x1F3FB #t)
	 (#t #\xAC01 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\xAC01 #f #\x200D #t)
	 (#t #\xAC01 #f #\x0308 #f #\x200D #t)
	 (#t #\xAC01 #t #\x2764 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x2764 #t)
	 (#t #\xAC01 #t #\x1F466 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x1F466 #t)
	 (#t #\xAC01 #t #\x0378 #t)
	 (#t #\xAC01 #f #\x0308 #t #\x0378 #t)
	 (#t #\xAC01 #t #\xD800 #t)
	 (#t #\xAC01 #f #\x0308 #t #\xD800 #t)
	 (#t #\x1F1E6 #t #\x0020 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0020 #t)
	 (#t #\x1F1E6 #t #\x000D #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x000D #t)
	 (#t #\x1F1E6 #t #\x000A #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x000A #t)
	 (#t #\x1F1E6 #t #\x0001 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F1E6 #f #\x0300 #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F1E6 #t #\x0600 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0600 #t)
	 (#t #\x1F1E6 #f #\x0903 #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x0903 #t)
	 (#t #\x1F1E6 #t #\x1100 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1100 #t)
	 (#t #\x1F1E6 #t #\x1160 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1160 #t)
	 (#t #\x1F1E6 #t #\x11A8 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x1F1E6 #t #\xAC00 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x1F1E6 #t #\xAC01 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x1F1E6 #f #\x1F1E6 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1F1E6 #t #\x261D #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x261D #t)
	 (#t #\x1F1E6 #t #\x1F3FB #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1F1E6 #f #\x200D #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x200D #t)
	 (#t #\x1F1E6 #t #\x2764 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F1E6 #t #\x1F466 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F1E6 #t #\x0378 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0378 #t)
	 (#t #\x1F1E6 #t #\xD800 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\xD800 #t)
	 (#t #\x261D #t #\x0020 #t)
	 (#t #\x261D #f #\x0308 #t #\x0020 #t)
	 (#t #\x261D #t #\x000D #t)
	 (#t #\x261D #f #\x0308 #t #\x000D #t)
	 (#t #\x261D #t #\x000A #t)
	 (#t #\x261D #f #\x0308 #t #\x000A #t)
	 (#t #\x261D #t #\x0001 #t)
	 (#t #\x261D #f #\x0308 #t #\x0001 #t)
	 (#t #\x261D #f #\x0300 #t)
	 (#t #\x261D #f #\x0308 #f #\x0300 #t)
	 (#t #\x261D #t #\x0600 #t)
	 (#t #\x261D #f #\x0308 #t #\x0600 #t)
	 (#t #\x261D #f #\x0903 #t)
	 (#t #\x261D #f #\x0308 #f #\x0903 #t)
	 (#t #\x261D #t #\x1100 #t)
	 (#t #\x261D #f #\x0308 #t #\x1100 #t)
	 (#t #\x261D #t #\x1160 #t)
	 (#t #\x261D #f #\x0308 #t #\x1160 #t)
	 (#t #\x261D #t #\x11A8 #t)
	 (#t #\x261D #f #\x0308 #t #\x11A8 #t)
	 (#t #\x261D #t #\xAC00 #t)
	 (#t #\x261D #f #\x0308 #t #\xAC00 #t)
	 (#t #\x261D #t #\xAC01 #t)
	 (#t #\x261D #f #\x0308 #t #\xAC01 #t)
	 (#t #\x261D #t #\x1F1E6 #t)
	 (#t #\x261D #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x261D #t #\x261D #t)
	 (#t #\x261D #f #\x0308 #t #\x261D #t)
	 (#t #\x261D #f #\x1F3FB #t)
	 (#t #\x261D #f #\x0308 #f #\x1F3FB #t)
	 (#t #\x261D #f #\x200D #t)
	 (#t #\x261D #f #\x0308 #f #\x200D #t)
	 (#t #\x261D #t #\x2764 #t)
	 (#t #\x261D #f #\x0308 #t #\x2764 #t)
	 (#t #\x261D #t #\x1F466 #t)
	 (#t #\x261D #f #\x0308 #t #\x1F466 #t)
	 (#t #\x261D #t #\x0378 #t)
	 (#t #\x261D #f #\x0308 #t #\x0378 #t)
	 (#t #\x261D #t #\xD800 #t)
	 (#t #\x261D #f #\x0308 #t #\xD800 #t)
	 (#t #\x1F3FB #t #\x0020 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0020 #t)
	 (#t #\x1F3FB #t #\x000D #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x000D #t)
	 (#t #\x1F3FB #t #\x000A #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x000A #t)
	 (#t #\x1F3FB #t #\x0001 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F3FB #f #\x0300 #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F3FB #t #\x0600 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0600 #t)
	 (#t #\x1F3FB #f #\x0903 #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x0903 #t)
	 (#t #\x1F3FB #t #\x1100 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1100 #t)
	 (#t #\x1F3FB #t #\x1160 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1160 #t)
	 (#t #\x1F3FB #t #\x11A8 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x11A8 #t)
	 (#t #\x1F3FB #t #\xAC00 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\xAC00 #t)
	 (#t #\x1F3FB #t #\xAC01 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\xAC01 #t)
	 (#t #\x1F3FB #t #\x1F1E6 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1F3FB #t #\x261D #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x261D #t)
	 (#t #\x1F3FB #t #\x1F3FB #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1F3FB #f #\x200D #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x200D #t)
	 (#t #\x1F3FB #t #\x2764 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F3FB #t #\x1F466 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F3FB #t #\x0378 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0378 #t)
	 (#t #\x1F3FB #t #\xD800 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\xD800 #t)
	 (#t #\x200D #t #\x0020 #t)
	 (#t #\x200D #f #\x0308 #t #\x0020 #t)
	 (#t #\x200D #t #\x000D #t)
	 (#t #\x200D #f #\x0308 #t #\x000D #t)
	 (#t #\x200D #t #\x000A #t)
	 (#t #\x200D #f #\x0308 #t #\x000A #t)
	 (#t #\x200D #t #\x0001 #t)
	 (#t #\x200D #f #\x0308 #t #\x0001 #t)
	 (#t #\x200D #f #\x0300 #t)
	 (#t #\x200D #f #\x0308 #f #\x0300 #t)
	 (#t #\x200D #t #\x0600 #t)
	 (#t #\x200D #f #\x0308 #t #\x0600 #t)
	 (#t #\x200D #f #\x0903 #t)
	 (#t #\x200D #f #\x0308 #f #\x0903 #t)
	 (#t #\x200D #t #\x1100 #t)
	 (#t #\x200D #f #\x0308 #t #\x1100 #t)
	 (#t #\x200D #t #\x1160 #t)
	 (#t #\x200D #f #\x0308 #t #\x1160 #t)
	 (#t #\x200D #t #\x11A8 #t)
	 (#t #\x200D #f #\x0308 #t #\x11A8 #t)
	 (#t #\x200D #t #\xAC00 #t)
	 (#t #\x200D #f #\x0308 #t #\xAC00 #t)
	 (#t #\x200D #t #\xAC01 #t)
	 (#t #\x200D #f #\x0308 #t #\xAC01 #t)
	 (#t #\x200D #t #\x1F1E6 #t)
	 (#t #\x200D #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x200D #t #\x261D #t)
	 (#t #\x200D #f #\x0308 #t #\x261D #t)
	 (#t #\x200D #t #\x1F3FB #t)
	 (#t #\x200D #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x200D #f #\x200D #t)
	 (#t #\x200D #f #\x0308 #f #\x200D #t)
	 (#t #\x200D #f #\x2764 #t)
	 (#t #\x200D #f #\x0308 #t #\x2764 #t)
	 (#t #\x200D #f #\x1F466 #t)
	 (#t #\x200D #f #\x0308 #t #\x1F466 #t)
	 (#t #\x200D #t #\x0378 #t)
	 (#t #\x200D #f #\x0308 #t #\x0378 #t)
	 (#t #\x200D #t #\xD800 #t)
	 (#t #\x200D #f #\x0308 #t #\xD800 #t)
	 (#t #\x2764 #t #\x0020 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0020 #t)
	 (#t #\x2764 #t #\x000D #t)
	 (#t #\x2764 #f #\x0308 #t #\x000D #t)
	 (#t #\x2764 #t #\x000A #t)
	 (#t #\x2764 #f #\x0308 #t #\x000A #t)
	 (#t #\x2764 #t #\x0001 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0001 #t)
	 (#t #\x2764 #f #\x0300 #t)
	 (#t #\x2764 #f #\x0308 #f #\x0300 #t)
	 (#t #\x2764 #t #\x0600 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0600 #t)
	 (#t #\x2764 #f #\x0903 #t)
	 (#t #\x2764 #f #\x0308 #f #\x0903 #t)
	 (#t #\x2764 #t #\x1100 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1100 #t)
	 (#t #\x2764 #t #\x1160 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1160 #t)
	 (#t #\x2764 #t #\x11A8 #t)
	 (#t #\x2764 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x2764 #t #\xAC00 #t)
	 (#t #\x2764 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x2764 #t #\xAC01 #t)
	 (#t #\x2764 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x2764 #t #\x1F1E6 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x2764 #t #\x261D #t)
	 (#t #\x2764 #f #\x0308 #t #\x261D #t)
	 (#t #\x2764 #t #\x1F3FB #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x2764 #f #\x200D #t)
	 (#t #\x2764 #f #\x0308 #f #\x200D #t)
	 (#t #\x2764 #t #\x2764 #t)
	 (#t #\x2764 #f #\x0308 #t #\x2764 #t)
	 (#t #\x2764 #t #\x1F466 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x2764 #t #\x0378 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0378 #t)
	 (#t #\x2764 #t #\xD800 #t)
	 (#t #\x2764 #f #\x0308 #t #\xD800 #t)
	 (#t #\x1F466 #t #\x0020 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0020 #t)
	 (#t #\x1F466 #t #\x000D #t)
	 (#t #\x1F466 #f #\x0308 #t #\x000D #t)
	 (#t #\x1F466 #t #\x000A #t)
	 (#t #\x1F466 #f #\x0308 #t #\x000A #t)
	 (#t #\x1F466 #t #\x0001 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F466 #f #\x0300 #t)
	 (#t #\x1F466 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F466 #t #\x0600 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0600 #t)
	 (#t #\x1F466 #f #\x0903 #t)
	 (#t #\x1F466 #f #\x0308 #f #\x0903 #t)
	 (#t #\x1F466 #t #\x1100 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1100 #t)
	 (#t #\x1F466 #t #\x1160 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1160 #t)
	 (#t #\x1F466 #t #\x11A8 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x1F466 #t #\xAC00 #t)
	 (#t #\x1F466 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x1F466 #t #\xAC01 #t)
	 (#t #\x1F466 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x1F466 #t #\x1F1E6 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1F466 #t #\x261D #t)
	 (#t #\x1F466 #f #\x0308 #t #\x261D #t)
	 (#t #\x1F466 #f #\x1F3FB #t)
	 (#t #\x1F466 #f #\x0308 #f #\x1F3FB #t)
	 (#t #\x1F466 #f #\x200D #t)
	 (#t #\x1F466 #f #\x0308 #f #\x200D #t)
	 (#t #\x1F466 #t #\x2764 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F466 #t #\x1F466 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F466 #t #\x0378 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0378 #t)
	 (#t #\x1F466 #t #\xD800 #t)
	 (#t #\x1F466 #f #\x0308 #t #\xD800 #t)
	 (#t #\x0378 #t #\x0020 #t)
	 (#t #\x0378 #f #\x0308 #t #\x0020 #t)
	 (#t #\x0378 #t #\x000D #t)
	 (#t #\x0378 #f #\x0308 #t #\x000D #t)
	 (#t #\x0378 #t #\x000A #t)
	 (#t #\x0378 #f #\x0308 #t #\x000A #t)
	 (#t #\x0378 #t #\x0001 #t)
	 (#t #\x0378 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0378 #f #\x0300 #t)
	 (#t #\x0378 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0378 #t #\x0600 #t)
	 (#t #\x0378 #f #\x0308 #t #\x0600 #t)
	 (#t #\x0378 #f #\x0903 #t)
	 (#t #\x0378 #f #\x0308 #f #\x0903 #t)
	 (#t #\x0378 #t #\x1100 #t)
	 (#t #\x0378 #f #\x0308 #t #\x1100 #t)
	 (#t #\x0378 #t #\x1160 #t)
	 (#t #\x0378 #f #\x0308 #t #\x1160 #t)
	 (#t #\x0378 #t #\x11A8 #t)
	 (#t #\x0378 #f #\x0308 #t #\x11A8 #t)
	 (#t #\x0378 #t #\xAC00 #t)
	 (#t #\x0378 #f #\x0308 #t #\xAC00 #t)
	 (#t #\x0378 #t #\xAC01 #t)
	 (#t #\x0378 #f #\x0308 #t #\xAC01 #t)
	 (#t #\x0378 #t #\x1F1E6 #t)
	 (#t #\x0378 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0378 #t #\x261D #t)
	 (#t #\x0378 #f #\x0308 #t #\x261D #t)
	 (#t #\x0378 #t #\x1F3FB #t)
	 (#t #\x0378 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0378 #f #\x200D #t)
	 (#t #\x0378 #f #\x0308 #f #\x200D #t)
	 (#t #\x0378 #t #\x2764 #t)
	 (#t #\x0378 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0378 #t #\x1F466 #t)
	 (#t #\x0378 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0378 #t #\x0378 #t)
	 (#t #\x0378 #f #\x0308 #t #\x0378 #t)
	 (#t #\x0378 #t #\xD800 #t)
	 (#t #\x0378 #f #\x0308 #t #\xD800 #t)
	 (#t #\xD800 #t #\x0020 #t)
	 (#t #\xD800 #t #\x0308 #t #\x0020 #t)
	 (#t #\xD800 #t #\x000D #t)
	 (#t #\xD800 #t #\x0308 #t #\x000D #t)
	 (#t #\xD800 #t #\x000A #t)
	 (#t #\xD800 #t #\x0308 #t #\x000A #t)
	 (#t #\xD800 #t #\x0001 #t)
	 (#t #\xD800 #t #\x0308 #t #\x0001 #t)
	 (#t #\xD800 #t #\x0300 #t)
	 (#t #\xD800 #t #\x0308 #f #\x0300 #t)
	 (#t #\xD800 #t #\x0600 #t)
	 (#t #\xD800 #t #\x0308 #t #\x0600 #t)
	 (#t #\xD800 #t #\x0903 #t)
	 (#t #\xD800 #t #\x0308 #f #\x0903 #t)
	 (#t #\xD800 #t #\x1100 #t)
	 (#t #\xD800 #t #\x0308 #t #\x1100 #t)
	 (#t #\xD800 #t #\x1160 #t)
	 (#t #\xD800 #t #\x0308 #t #\x1160 #t)
	 (#t #\xD800 #t #\x11A8 #t)
	 (#t #\xD800 #t #\x0308 #t #\x11A8 #t)
	 (#t #\xD800 #t #\xAC00 #t)
	 (#t #\xD800 #t #\x0308 #t #\xAC00 #t)
	 (#t #\xD800 #t #\xAC01 #t)
	 (#t #\xD800 #t #\x0308 #t #\xAC01 #t)
	 (#t #\xD800 #t #\x1F1E6 #t)
	 (#t #\xD800 #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\xD800 #t #\x261D #t)
	 (#t #\xD800 #t #\x0308 #t #\x261D #t)
	 (#t #\xD800 #t #\x1F3FB #t)
	 (#t #\xD800 #t #\x0308 #t #\x1F3FB #t)
	 (#t #\xD800 #t #\x200D #t)
	 (#t #\xD800 #t #\x0308 #f #\x200D #t)
	 (#t #\xD800 #t #\x2764 #t)
	 (#t #\xD800 #t #\x0308 #t #\x2764 #t)
	 (#t #\xD800 #t #\x1F466 #t)
	 (#t #\xD800 #t #\x0308 #t #\x1F466 #t)
	 (#t #\xD800 #t #\x0378 #t)
	 (#t #\xD800 #t #\x0308 #t #\x0378 #t)
	 (#t #\xD800 #t #\xD800 #t)
	 (#t #\xD800 #t #\x0308 #t #\xD800 #t)
	 (#t #\x000D #f #\x000A #t #\x0061 #t #\x000A #t #\x0308 #t)
	 (#t #\x0061 #f #\x0308 #t)
	 (#t #\x0020 #f #\x200D #t #\x0646 #t)
	 (#t #\x0646 #f #\x200D #t #\x0020 #t)
	 (#t #\x1100 #f #\x1100 #t)
	 (#t #\xAC00 #f #\x11A8 #t #\x1100 #t)
	 (#t #\xAC01 #f #\x11A8 #t #\x1100 #t)
	 (#t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #f #\x200D #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x200D #t #\x1F1E7 #f #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #f #\x1F1E9 #t #\x0062 #t)
	 (#t #\x0061 #f #\x200D #t)
	 (#t #\x0061 #f #\x0308 #t #\x0062 #t)
	 (#t #\x0061 #f #\x0903 #t #\x0062 #t)
	 (#t #\x0061 #t #\x0600 #f #\x0062 #t)
	 (#t #\x261D #f #\x1F3FB #t #\x261D #t)
	 (#t #\x1F466 #f #\x1F3FB #t)
	 (#t #\x200D #f #\x1F466 #f #\x1F3FB #t)
	 (#t #\x200D #f #\x2764 #t)
	 (#t #\x200D #f #\x1F466 #t)
	 (#t #\x1F466 #t #\x1F466 #t))))

(define-test 'word-breaks
  (map (lambda (test-case)
	 (lambda ()
	   (run-break-test string-word-breaks test-case)))
       '((#t #\x0001 #t #\x0001 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0001 #t #\x000D #t)
	 (#t #\x0001 #f #\x0308 #t #\x000D #t)
	 (#t #\x0001 #t #\x000A #t)
	 (#t #\x0001 #f #\x0308 #t #\x000A #t)
	 (#t #\x0001 #t #\x000B #t)
	 (#t #\x0001 #f #\x0308 #t #\x000B #t)
	 (#t #\x0001 #t #\x3031 #t)
	 (#t #\x0001 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0001 #t #\x0041 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0001 #t #\x003A #t)
	 (#t #\x0001 #f #\x0308 #t #\x003A #t)
	 (#t #\x0001 #t #\x002C #t)
	 (#t #\x0001 #f #\x0308 #t #\x002C #t)
	 (#t #\x0001 #t #\x002E #t)
	 (#t #\x0001 #f #\x0308 #t #\x002E #t)
	 (#t #\x0001 #t #\x0030 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0001 #t #\x005F #t)
	 (#t #\x0001 #f #\x0308 #t #\x005F #t)
	 (#t #\x0001 #t #\x1F1E6 #t)
	 (#t #\x0001 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0001 #t #\x05D0 #t)
	 (#t #\x0001 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0001 #t #\x0022 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0001 #t #\x0027 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0001 #t #\x261D #t)
	 (#t #\x0001 #f #\x0308 #t #\x261D #t)
	 (#t #\x0001 #t #\x1F3FB #t)
	 (#t #\x0001 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0001 #t #\x2764 #t)
	 (#t #\x0001 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0001 #t #\x1F466 #t)
	 (#t #\x0001 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0001 #f #\x00AD #t)
	 (#t #\x0001 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0001 #f #\x0300 #t)
	 (#t #\x0001 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0001 #f #\x200D #t)
	 (#t #\x0001 #f #\x0308 #f #\x200D #t)
	 (#t #\x0001 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0001 #t #\x0061 #t #\x003A #t)
	 (#t #\x0001 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0001 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0001 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0001 #t #\x0061 #t #\x002C #t)
	 (#t #\x0001 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0001 #t #\x0031 #t #\x003A #t)
	 (#t #\x0001 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0001 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0001 #t #\x0031 #t #\x002C #t)
	 (#t #\x0001 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0001 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0001 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000D #t #\x0001 #t)
	 (#t #\x000D #t #\x0308 #t #\x0001 #t)
	 (#t #\x000D #t #\x000D #t)
	 (#t #\x000D #t #\x0308 #t #\x000D #t)
	 (#t #\x000D #f #\x000A #t)
	 (#t #\x000D #t #\x0308 #t #\x000A #t)
	 (#t #\x000D #t #\x000B #t)
	 (#t #\x000D #t #\x0308 #t #\x000B #t)
	 (#t #\x000D #t #\x3031 #t)
	 (#t #\x000D #t #\x0308 #t #\x3031 #t)
	 (#t #\x000D #t #\x0041 #t)
	 (#t #\x000D #t #\x0308 #t #\x0041 #t)
	 (#t #\x000D #t #\x003A #t)
	 (#t #\x000D #t #\x0308 #t #\x003A #t)
	 (#t #\x000D #t #\x002C #t)
	 (#t #\x000D #t #\x0308 #t #\x002C #t)
	 (#t #\x000D #t #\x002E #t)
	 (#t #\x000D #t #\x0308 #t #\x002E #t)
	 (#t #\x000D #t #\x0030 #t)
	 (#t #\x000D #t #\x0308 #t #\x0030 #t)
	 (#t #\x000D #t #\x005F #t)
	 (#t #\x000D #t #\x0308 #t #\x005F #t)
	 (#t #\x000D #t #\x1F1E6 #t)
	 (#t #\x000D #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x000D #t #\x05D0 #t)
	 (#t #\x000D #t #\x0308 #t #\x05D0 #t)
	 (#t #\x000D #t #\x0022 #t)
	 (#t #\x000D #t #\x0308 #t #\x0022 #t)
	 (#t #\x000D #t #\x0027 #t)
	 (#t #\x000D #t #\x0308 #t #\x0027 #t)
	 (#t #\x000D #t #\x261D #t)
	 (#t #\x000D #t #\x0308 #t #\x261D #t)
	 (#t #\x000D #t #\x1F3FB #t)
	 (#t #\x000D #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x000D #t #\x2764 #t)
	 (#t #\x000D #t #\x0308 #t #\x2764 #t)
	 (#t #\x000D #t #\x1F466 #t)
	 (#t #\x000D #t #\x0308 #t #\x1F466 #t)
	 (#t #\x000D #t #\x00AD #t)
	 (#t #\x000D #t #\x0308 #f #\x00AD #t)
	 (#t #\x000D #t #\x0300 #t)
	 (#t #\x000D #t #\x0308 #f #\x0300 #t)
	 (#t #\x000D #t #\x200D #t)
	 (#t #\x000D #t #\x0308 #f #\x200D #t)
	 (#t #\x000D #t #\x0061 #f #\x2060 #t)
	 (#t #\x000D #t #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x000D #t #\x0061 #t #\x003A #t)
	 (#t #\x000D #t #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x000D #t #\x0061 #t #\x0027 #t)
	 (#t #\x000D #t #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x000D #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000D #t #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000D #t #\x0061 #t #\x002C #t)
	 (#t #\x000D #t #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x000D #t #\x0031 #t #\x003A #t)
	 (#t #\x000D #t #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x000D #t #\x0031 #t #\x0027 #t)
	 (#t #\x000D #t #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x000D #t #\x0031 #t #\x002C #t)
	 (#t #\x000D #t #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x000D #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000D #t #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000A #t #\x0001 #t)
	 (#t #\x000A #t #\x0308 #t #\x0001 #t)
	 (#t #\x000A #t #\x000D #t)
	 (#t #\x000A #t #\x0308 #t #\x000D #t)
	 (#t #\x000A #t #\x000A #t)
	 (#t #\x000A #t #\x0308 #t #\x000A #t)
	 (#t #\x000A #t #\x000B #t)
	 (#t #\x000A #t #\x0308 #t #\x000B #t)
	 (#t #\x000A #t #\x3031 #t)
	 (#t #\x000A #t #\x0308 #t #\x3031 #t)
	 (#t #\x000A #t #\x0041 #t)
	 (#t #\x000A #t #\x0308 #t #\x0041 #t)
	 (#t #\x000A #t #\x003A #t)
	 (#t #\x000A #t #\x0308 #t #\x003A #t)
	 (#t #\x000A #t #\x002C #t)
	 (#t #\x000A #t #\x0308 #t #\x002C #t)
	 (#t #\x000A #t #\x002E #t)
	 (#t #\x000A #t #\x0308 #t #\x002E #t)
	 (#t #\x000A #t #\x0030 #t)
	 (#t #\x000A #t #\x0308 #t #\x0030 #t)
	 (#t #\x000A #t #\x005F #t)
	 (#t #\x000A #t #\x0308 #t #\x005F #t)
	 (#t #\x000A #t #\x1F1E6 #t)
	 (#t #\x000A #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x000A #t #\x05D0 #t)
	 (#t #\x000A #t #\x0308 #t #\x05D0 #t)
	 (#t #\x000A #t #\x0022 #t)
	 (#t #\x000A #t #\x0308 #t #\x0022 #t)
	 (#t #\x000A #t #\x0027 #t)
	 (#t #\x000A #t #\x0308 #t #\x0027 #t)
	 (#t #\x000A #t #\x261D #t)
	 (#t #\x000A #t #\x0308 #t #\x261D #t)
	 (#t #\x000A #t #\x1F3FB #t)
	 (#t #\x000A #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x000A #t #\x2764 #t)
	 (#t #\x000A #t #\x0308 #t #\x2764 #t)
	 (#t #\x000A #t #\x1F466 #t)
	 (#t #\x000A #t #\x0308 #t #\x1F466 #t)
	 (#t #\x000A #t #\x00AD #t)
	 (#t #\x000A #t #\x0308 #f #\x00AD #t)
	 (#t #\x000A #t #\x0300 #t)
	 (#t #\x000A #t #\x0308 #f #\x0300 #t)
	 (#t #\x000A #t #\x200D #t)
	 (#t #\x000A #t #\x0308 #f #\x200D #t)
	 (#t #\x000A #t #\x0061 #f #\x2060 #t)
	 (#t #\x000A #t #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x000A #t #\x0061 #t #\x003A #t)
	 (#t #\x000A #t #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x000A #t #\x0061 #t #\x0027 #t)
	 (#t #\x000A #t #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x000A #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000A #t #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000A #t #\x0061 #t #\x002C #t)
	 (#t #\x000A #t #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x000A #t #\x0031 #t #\x003A #t)
	 (#t #\x000A #t #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x000A #t #\x0031 #t #\x0027 #t)
	 (#t #\x000A #t #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x000A #t #\x0031 #t #\x002C #t)
	 (#t #\x000A #t #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x000A #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000A #t #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000B #t #\x0001 #t)
	 (#t #\x000B #t #\x0308 #t #\x0001 #t)
	 (#t #\x000B #t #\x000D #t)
	 (#t #\x000B #t #\x0308 #t #\x000D #t)
	 (#t #\x000B #t #\x000A #t)
	 (#t #\x000B #t #\x0308 #t #\x000A #t)
	 (#t #\x000B #t #\x000B #t)
	 (#t #\x000B #t #\x0308 #t #\x000B #t)
	 (#t #\x000B #t #\x3031 #t)
	 (#t #\x000B #t #\x0308 #t #\x3031 #t)
	 (#t #\x000B #t #\x0041 #t)
	 (#t #\x000B #t #\x0308 #t #\x0041 #t)
	 (#t #\x000B #t #\x003A #t)
	 (#t #\x000B #t #\x0308 #t #\x003A #t)
	 (#t #\x000B #t #\x002C #t)
	 (#t #\x000B #t #\x0308 #t #\x002C #t)
	 (#t #\x000B #t #\x002E #t)
	 (#t #\x000B #t #\x0308 #t #\x002E #t)
	 (#t #\x000B #t #\x0030 #t)
	 (#t #\x000B #t #\x0308 #t #\x0030 #t)
	 (#t #\x000B #t #\x005F #t)
	 (#t #\x000B #t #\x0308 #t #\x005F #t)
	 (#t #\x000B #t #\x1F1E6 #t)
	 (#t #\x000B #t #\x0308 #t #\x1F1E6 #t)
	 (#t #\x000B #t #\x05D0 #t)
	 (#t #\x000B #t #\x0308 #t #\x05D0 #t)
	 (#t #\x000B #t #\x0022 #t)
	 (#t #\x000B #t #\x0308 #t #\x0022 #t)
	 (#t #\x000B #t #\x0027 #t)
	 (#t #\x000B #t #\x0308 #t #\x0027 #t)
	 (#t #\x000B #t #\x261D #t)
	 (#t #\x000B #t #\x0308 #t #\x261D #t)
	 (#t #\x000B #t #\x1F3FB #t)
	 (#t #\x000B #t #\x0308 #t #\x1F3FB #t)
	 (#t #\x000B #t #\x2764 #t)
	 (#t #\x000B #t #\x0308 #t #\x2764 #t)
	 (#t #\x000B #t #\x1F466 #t)
	 (#t #\x000B #t #\x0308 #t #\x1F466 #t)
	 (#t #\x000B #t #\x00AD #t)
	 (#t #\x000B #t #\x0308 #f #\x00AD #t)
	 (#t #\x000B #t #\x0300 #t)
	 (#t #\x000B #t #\x0308 #f #\x0300 #t)
	 (#t #\x000B #t #\x200D #t)
	 (#t #\x000B #t #\x0308 #f #\x200D #t)
	 (#t #\x000B #t #\x0061 #f #\x2060 #t)
	 (#t #\x000B #t #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x000B #t #\x0061 #t #\x003A #t)
	 (#t #\x000B #t #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x000B #t #\x0061 #t #\x0027 #t)
	 (#t #\x000B #t #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x000B #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000B #t #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x000B #t #\x0061 #t #\x002C #t)
	 (#t #\x000B #t #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x000B #t #\x0031 #t #\x003A #t)
	 (#t #\x000B #t #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x000B #t #\x0031 #t #\x0027 #t)
	 (#t #\x000B #t #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x000B #t #\x0031 #t #\x002C #t)
	 (#t #\x000B #t #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x000B #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000B #t #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x3031 #t #\x0001 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0001 #t)
	 (#t #\x3031 #t #\x000D #t)
	 (#t #\x3031 #f #\x0308 #t #\x000D #t)
	 (#t #\x3031 #t #\x000A #t)
	 (#t #\x3031 #f #\x0308 #t #\x000A #t)
	 (#t #\x3031 #t #\x000B #t)
	 (#t #\x3031 #f #\x0308 #t #\x000B #t)
	 (#t #\x3031 #f #\x3031 #t)
	 (#t #\x3031 #f #\x0308 #f #\x3031 #t)
	 (#t #\x3031 #t #\x0041 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0041 #t)
	 (#t #\x3031 #t #\x003A #t)
	 (#t #\x3031 #f #\x0308 #t #\x003A #t)
	 (#t #\x3031 #t #\x002C #t)
	 (#t #\x3031 #f #\x0308 #t #\x002C #t)
	 (#t #\x3031 #t #\x002E #t)
	 (#t #\x3031 #f #\x0308 #t #\x002E #t)
	 (#t #\x3031 #t #\x0030 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0030 #t)
	 (#t #\x3031 #f #\x005F #t)
	 (#t #\x3031 #f #\x0308 #f #\x005F #t)
	 (#t #\x3031 #t #\x1F1E6 #t)
	 (#t #\x3031 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x3031 #t #\x05D0 #t)
	 (#t #\x3031 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x3031 #t #\x0022 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0022 #t)
	 (#t #\x3031 #t #\x0027 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0027 #t)
	 (#t #\x3031 #t #\x261D #t)
	 (#t #\x3031 #f #\x0308 #t #\x261D #t)
	 (#t #\x3031 #t #\x1F3FB #t)
	 (#t #\x3031 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x3031 #t #\x2764 #t)
	 (#t #\x3031 #f #\x0308 #t #\x2764 #t)
	 (#t #\x3031 #t #\x1F466 #t)
	 (#t #\x3031 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x3031 #f #\x00AD #t)
	 (#t #\x3031 #f #\x0308 #f #\x00AD #t)
	 (#t #\x3031 #f #\x0300 #t)
	 (#t #\x3031 #f #\x0308 #f #\x0300 #t)
	 (#t #\x3031 #f #\x200D #t)
	 (#t #\x3031 #f #\x0308 #f #\x200D #t)
	 (#t #\x3031 #t #\x0061 #f #\x2060 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x3031 #t #\x0061 #t #\x003A #t)
	 (#t #\x3031 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x3031 #t #\x0061 #t #\x0027 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x3031 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x3031 #t #\x0061 #t #\x002C #t)
	 (#t #\x3031 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x3031 #t #\x0031 #t #\x003A #t)
	 (#t #\x3031 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x3031 #t #\x0031 #t #\x0027 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x3031 #t #\x0031 #t #\x002C #t)
	 (#t #\x3031 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x3031 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x3031 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0041 #t #\x0001 #t)
	 (#t #\x0041 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0041 #t #\x000D #t)
	 (#t #\x0041 #f #\x0308 #t #\x000D #t)
	 (#t #\x0041 #t #\x000A #t)
	 (#t #\x0041 #f #\x0308 #t #\x000A #t)
	 (#t #\x0041 #t #\x000B #t)
	 (#t #\x0041 #f #\x0308 #t #\x000B #t)
	 (#t #\x0041 #t #\x3031 #t)
	 (#t #\x0041 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0041 #f #\x0041 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0041 #t)
	 (#t #\x0041 #t #\x003A #t)
	 (#t #\x0041 #f #\x0308 #t #\x003A #t)
	 (#t #\x0041 #t #\x002C #t)
	 (#t #\x0041 #f #\x0308 #t #\x002C #t)
	 (#t #\x0041 #t #\x002E #t)
	 (#t #\x0041 #f #\x0308 #t #\x002E #t)
	 (#t #\x0041 #f #\x0030 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0030 #t)
	 (#t #\x0041 #f #\x005F #t)
	 (#t #\x0041 #f #\x0308 #f #\x005F #t)
	 (#t #\x0041 #t #\x1F1E6 #t)
	 (#t #\x0041 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0041 #f #\x05D0 #t)
	 (#t #\x0041 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0041 #t #\x0022 #t)
	 (#t #\x0041 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0041 #t #\x0027 #t)
	 (#t #\x0041 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0041 #t #\x261D #t)
	 (#t #\x0041 #f #\x0308 #t #\x261D #t)
	 (#t #\x0041 #t #\x1F3FB #t)
	 (#t #\x0041 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0041 #t #\x2764 #t)
	 (#t #\x0041 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0041 #t #\x1F466 #t)
	 (#t #\x0041 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0041 #f #\x00AD #t)
	 (#t #\x0041 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0041 #f #\x0300 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0041 #f #\x200D #t)
	 (#t #\x0041 #f #\x0308 #f #\x200D #t)
	 (#t #\x0041 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0041 #f #\x0061 #t #\x003A #t)
	 (#t #\x0041 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0041 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0041 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0041 #f #\x0061 #t #\x002C #t)
	 (#t #\x0041 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0041 #f #\x0031 #t #\x003A #t)
	 (#t #\x0041 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0041 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0041 #f #\x0031 #t #\x002C #t)
	 (#t #\x0041 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0041 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0041 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x003A #t #\x0001 #t)
	 (#t #\x003A #f #\x0308 #t #\x0001 #t)
	 (#t #\x003A #t #\x000D #t)
	 (#t #\x003A #f #\x0308 #t #\x000D #t)
	 (#t #\x003A #t #\x000A #t)
	 (#t #\x003A #f #\x0308 #t #\x000A #t)
	 (#t #\x003A #t #\x000B #t)
	 (#t #\x003A #f #\x0308 #t #\x000B #t)
	 (#t #\x003A #t #\x3031 #t)
	 (#t #\x003A #f #\x0308 #t #\x3031 #t)
	 (#t #\x003A #t #\x0041 #t)
	 (#t #\x003A #f #\x0308 #t #\x0041 #t)
	 (#t #\x003A #t #\x003A #t)
	 (#t #\x003A #f #\x0308 #t #\x003A #t)
	 (#t #\x003A #t #\x002C #t)
	 (#t #\x003A #f #\x0308 #t #\x002C #t)
	 (#t #\x003A #t #\x002E #t)
	 (#t #\x003A #f #\x0308 #t #\x002E #t)
	 (#t #\x003A #t #\x0030 #t)
	 (#t #\x003A #f #\x0308 #t #\x0030 #t)
	 (#t #\x003A #t #\x005F #t)
	 (#t #\x003A #f #\x0308 #t #\x005F #t)
	 (#t #\x003A #t #\x1F1E6 #t)
	 (#t #\x003A #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x003A #t #\x05D0 #t)
	 (#t #\x003A #f #\x0308 #t #\x05D0 #t)
	 (#t #\x003A #t #\x0022 #t)
	 (#t #\x003A #f #\x0308 #t #\x0022 #t)
	 (#t #\x003A #t #\x0027 #t)
	 (#t #\x003A #f #\x0308 #t #\x0027 #t)
	 (#t #\x003A #t #\x261D #t)
	 (#t #\x003A #f #\x0308 #t #\x261D #t)
	 (#t #\x003A #t #\x1F3FB #t)
	 (#t #\x003A #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x003A #t #\x2764 #t)
	 (#t #\x003A #f #\x0308 #t #\x2764 #t)
	 (#t #\x003A #t #\x1F466 #t)
	 (#t #\x003A #f #\x0308 #t #\x1F466 #t)
	 (#t #\x003A #f #\x00AD #t)
	 (#t #\x003A #f #\x0308 #f #\x00AD #t)
	 (#t #\x003A #f #\x0300 #t)
	 (#t #\x003A #f #\x0308 #f #\x0300 #t)
	 (#t #\x003A #f #\x200D #t)
	 (#t #\x003A #f #\x0308 #f #\x200D #t)
	 (#t #\x003A #t #\x0061 #f #\x2060 #t)
	 (#t #\x003A #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x003A #t #\x0061 #t #\x003A #t)
	 (#t #\x003A #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x003A #t #\x0061 #t #\x0027 #t)
	 (#t #\x003A #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x003A #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x003A #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x003A #t #\x0061 #t #\x002C #t)
	 (#t #\x003A #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x003A #t #\x0031 #t #\x003A #t)
	 (#t #\x003A #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x003A #t #\x0031 #t #\x0027 #t)
	 (#t #\x003A #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x003A #t #\x0031 #t #\x002C #t)
	 (#t #\x003A #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x003A #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x003A #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x002C #t #\x0001 #t)
	 (#t #\x002C #f #\x0308 #t #\x0001 #t)
	 (#t #\x002C #t #\x000D #t)
	 (#t #\x002C #f #\x0308 #t #\x000D #t)
	 (#t #\x002C #t #\x000A #t)
	 (#t #\x002C #f #\x0308 #t #\x000A #t)
	 (#t #\x002C #t #\x000B #t)
	 (#t #\x002C #f #\x0308 #t #\x000B #t)
	 (#t #\x002C #t #\x3031 #t)
	 (#t #\x002C #f #\x0308 #t #\x3031 #t)
	 (#t #\x002C #t #\x0041 #t)
	 (#t #\x002C #f #\x0308 #t #\x0041 #t)
	 (#t #\x002C #t #\x003A #t)
	 (#t #\x002C #f #\x0308 #t #\x003A #t)
	 (#t #\x002C #t #\x002C #t)
	 (#t #\x002C #f #\x0308 #t #\x002C #t)
	 (#t #\x002C #t #\x002E #t)
	 (#t #\x002C #f #\x0308 #t #\x002E #t)
	 (#t #\x002C #t #\x0030 #t)
	 (#t #\x002C #f #\x0308 #t #\x0030 #t)
	 (#t #\x002C #t #\x005F #t)
	 (#t #\x002C #f #\x0308 #t #\x005F #t)
	 (#t #\x002C #t #\x1F1E6 #t)
	 (#t #\x002C #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x002C #t #\x05D0 #t)
	 (#t #\x002C #f #\x0308 #t #\x05D0 #t)
	 (#t #\x002C #t #\x0022 #t)
	 (#t #\x002C #f #\x0308 #t #\x0022 #t)
	 (#t #\x002C #t #\x0027 #t)
	 (#t #\x002C #f #\x0308 #t #\x0027 #t)
	 (#t #\x002C #t #\x261D #t)
	 (#t #\x002C #f #\x0308 #t #\x261D #t)
	 (#t #\x002C #t #\x1F3FB #t)
	 (#t #\x002C #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x002C #t #\x2764 #t)
	 (#t #\x002C #f #\x0308 #t #\x2764 #t)
	 (#t #\x002C #t #\x1F466 #t)
	 (#t #\x002C #f #\x0308 #t #\x1F466 #t)
	 (#t #\x002C #f #\x00AD #t)
	 (#t #\x002C #f #\x0308 #f #\x00AD #t)
	 (#t #\x002C #f #\x0300 #t)
	 (#t #\x002C #f #\x0308 #f #\x0300 #t)
	 (#t #\x002C #f #\x200D #t)
	 (#t #\x002C #f #\x0308 #f #\x200D #t)
	 (#t #\x002C #t #\x0061 #f #\x2060 #t)
	 (#t #\x002C #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x002C #t #\x0061 #t #\x003A #t)
	 (#t #\x002C #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x002C #t #\x0061 #t #\x0027 #t)
	 (#t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x002C #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x002C #t #\x0061 #t #\x002C #t)
	 (#t #\x002C #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x002C #t #\x0031 #t #\x003A #t)
	 (#t #\x002C #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x002C #t #\x0031 #t #\x0027 #t)
	 (#t #\x002C #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x002C #t #\x0031 #t #\x002C #t)
	 (#t #\x002C #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x002C #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x002C #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x002E #t #\x0001 #t)
	 (#t #\x002E #f #\x0308 #t #\x0001 #t)
	 (#t #\x002E #t #\x000D #t)
	 (#t #\x002E #f #\x0308 #t #\x000D #t)
	 (#t #\x002E #t #\x000A #t)
	 (#t #\x002E #f #\x0308 #t #\x000A #t)
	 (#t #\x002E #t #\x000B #t)
	 (#t #\x002E #f #\x0308 #t #\x000B #t)
	 (#t #\x002E #t #\x3031 #t)
	 (#t #\x002E #f #\x0308 #t #\x3031 #t)
	 (#t #\x002E #t #\x0041 #t)
	 (#t #\x002E #f #\x0308 #t #\x0041 #t)
	 (#t #\x002E #t #\x003A #t)
	 (#t #\x002E #f #\x0308 #t #\x003A #t)
	 (#t #\x002E #t #\x002C #t)
	 (#t #\x002E #f #\x0308 #t #\x002C #t)
	 (#t #\x002E #t #\x002E #t)
	 (#t #\x002E #f #\x0308 #t #\x002E #t)
	 (#t #\x002E #t #\x0030 #t)
	 (#t #\x002E #f #\x0308 #t #\x0030 #t)
	 (#t #\x002E #t #\x005F #t)
	 (#t #\x002E #f #\x0308 #t #\x005F #t)
	 (#t #\x002E #t #\x1F1E6 #t)
	 (#t #\x002E #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x002E #t #\x05D0 #t)
	 (#t #\x002E #f #\x0308 #t #\x05D0 #t)
	 (#t #\x002E #t #\x0022 #t)
	 (#t #\x002E #f #\x0308 #t #\x0022 #t)
	 (#t #\x002E #t #\x0027 #t)
	 (#t #\x002E #f #\x0308 #t #\x0027 #t)
	 (#t #\x002E #t #\x261D #t)
	 (#t #\x002E #f #\x0308 #t #\x261D #t)
	 (#t #\x002E #t #\x1F3FB #t)
	 (#t #\x002E #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x002E #t #\x2764 #t)
	 (#t #\x002E #f #\x0308 #t #\x2764 #t)
	 (#t #\x002E #t #\x1F466 #t)
	 (#t #\x002E #f #\x0308 #t #\x1F466 #t)
	 (#t #\x002E #f #\x00AD #t)
	 (#t #\x002E #f #\x0308 #f #\x00AD #t)
	 (#t #\x002E #f #\x0300 #t)
	 (#t #\x002E #f #\x0308 #f #\x0300 #t)
	 (#t #\x002E #f #\x200D #t)
	 (#t #\x002E #f #\x0308 #f #\x200D #t)
	 (#t #\x002E #t #\x0061 #f #\x2060 #t)
	 (#t #\x002E #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x002E #t #\x0061 #t #\x003A #t)
	 (#t #\x002E #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x002E #t #\x0061 #t #\x0027 #t)
	 (#t #\x002E #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x002E #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x002E #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x002E #t #\x0061 #t #\x002C #t)
	 (#t #\x002E #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x002E #t #\x0031 #t #\x003A #t)
	 (#t #\x002E #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x002E #t #\x0031 #t #\x0027 #t)
	 (#t #\x002E #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x002E #t #\x0031 #t #\x002C #t)
	 (#t #\x002E #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x002E #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x002E #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0030 #t #\x0001 #t)
	 (#t #\x0030 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0030 #t #\x000D #t)
	 (#t #\x0030 #f #\x0308 #t #\x000D #t)
	 (#t #\x0030 #t #\x000A #t)
	 (#t #\x0030 #f #\x0308 #t #\x000A #t)
	 (#t #\x0030 #t #\x000B #t)
	 (#t #\x0030 #f #\x0308 #t #\x000B #t)
	 (#t #\x0030 #t #\x3031 #t)
	 (#t #\x0030 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0030 #f #\x0041 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0041 #t)
	 (#t #\x0030 #t #\x003A #t)
	 (#t #\x0030 #f #\x0308 #t #\x003A #t)
	 (#t #\x0030 #t #\x002C #t)
	 (#t #\x0030 #f #\x0308 #t #\x002C #t)
	 (#t #\x0030 #t #\x002E #t)
	 (#t #\x0030 #f #\x0308 #t #\x002E #t)
	 (#t #\x0030 #f #\x0030 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0030 #t)
	 (#t #\x0030 #f #\x005F #t)
	 (#t #\x0030 #f #\x0308 #f #\x005F #t)
	 (#t #\x0030 #t #\x1F1E6 #t)
	 (#t #\x0030 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0030 #f #\x05D0 #t)
	 (#t #\x0030 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0030 #t #\x0022 #t)
	 (#t #\x0030 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0030 #t #\x0027 #t)
	 (#t #\x0030 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0030 #t #\x261D #t)
	 (#t #\x0030 #f #\x0308 #t #\x261D #t)
	 (#t #\x0030 #t #\x1F3FB #t)
	 (#t #\x0030 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0030 #t #\x2764 #t)
	 (#t #\x0030 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0030 #t #\x1F466 #t)
	 (#t #\x0030 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0030 #f #\x00AD #t)
	 (#t #\x0030 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0030 #f #\x0300 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0030 #f #\x200D #t)
	 (#t #\x0030 #f #\x0308 #f #\x200D #t)
	 (#t #\x0030 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0030 #f #\x0061 #t #\x003A #t)
	 (#t #\x0030 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0030 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0030 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0030 #f #\x0061 #t #\x002C #t)
	 (#t #\x0030 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0030 #f #\x0031 #t #\x003A #t)
	 (#t #\x0030 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0030 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0030 #f #\x0031 #t #\x002C #t)
	 (#t #\x0030 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0030 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0030 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x005F #t #\x0001 #t)
	 (#t #\x005F #f #\x0308 #t #\x0001 #t)
	 (#t #\x005F #t #\x000D #t)
	 (#t #\x005F #f #\x0308 #t #\x000D #t)
	 (#t #\x005F #t #\x000A #t)
	 (#t #\x005F #f #\x0308 #t #\x000A #t)
	 (#t #\x005F #t #\x000B #t)
	 (#t #\x005F #f #\x0308 #t #\x000B #t)
	 (#t #\x005F #f #\x3031 #t)
	 (#t #\x005F #f #\x0308 #f #\x3031 #t)
	 (#t #\x005F #f #\x0041 #t)
	 (#t #\x005F #f #\x0308 #f #\x0041 #t)
	 (#t #\x005F #t #\x003A #t)
	 (#t #\x005F #f #\x0308 #t #\x003A #t)
	 (#t #\x005F #t #\x002C #t)
	 (#t #\x005F #f #\x0308 #t #\x002C #t)
	 (#t #\x005F #t #\x002E #t)
	 (#t #\x005F #f #\x0308 #t #\x002E #t)
	 (#t #\x005F #f #\x0030 #t)
	 (#t #\x005F #f #\x0308 #f #\x0030 #t)
	 (#t #\x005F #f #\x005F #t)
	 (#t #\x005F #f #\x0308 #f #\x005F #t)
	 (#t #\x005F #t #\x1F1E6 #t)
	 (#t #\x005F #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x005F #f #\x05D0 #t)
	 (#t #\x005F #f #\x0308 #f #\x05D0 #t)
	 (#t #\x005F #t #\x0022 #t)
	 (#t #\x005F #f #\x0308 #t #\x0022 #t)
	 (#t #\x005F #t #\x0027 #t)
	 (#t #\x005F #f #\x0308 #t #\x0027 #t)
	 (#t #\x005F #t #\x261D #t)
	 (#t #\x005F #f #\x0308 #t #\x261D #t)
	 (#t #\x005F #t #\x1F3FB #t)
	 (#t #\x005F #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x005F #t #\x2764 #t)
	 (#t #\x005F #f #\x0308 #t #\x2764 #t)
	 (#t #\x005F #t #\x1F466 #t)
	 (#t #\x005F #f #\x0308 #t #\x1F466 #t)
	 (#t #\x005F #f #\x00AD #t)
	 (#t #\x005F #f #\x0308 #f #\x00AD #t)
	 (#t #\x005F #f #\x0300 #t)
	 (#t #\x005F #f #\x0308 #f #\x0300 #t)
	 (#t #\x005F #f #\x200D #t)
	 (#t #\x005F #f #\x0308 #f #\x200D #t)
	 (#t #\x005F #f #\x0061 #f #\x2060 #t)
	 (#t #\x005F #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x005F #f #\x0061 #t #\x003A #t)
	 (#t #\x005F #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x005F #f #\x0061 #t #\x0027 #t)
	 (#t #\x005F #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x005F #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x005F #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x005F #f #\x0061 #t #\x002C #t)
	 (#t #\x005F #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x005F #f #\x0031 #t #\x003A #t)
	 (#t #\x005F #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x005F #f #\x0031 #t #\x0027 #t)
	 (#t #\x005F #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x005F #f #\x0031 #t #\x002C #t)
	 (#t #\x005F #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x005F #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x005F #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F1E6 #t #\x0001 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F1E6 #t #\x000D #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x000D #t)
	 (#t #\x1F1E6 #t #\x000A #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x000A #t)
	 (#t #\x1F1E6 #t #\x000B #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x000B #t)
	 (#t #\x1F1E6 #t #\x3031 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x3031 #t)
	 (#t #\x1F1E6 #t #\x0041 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0041 #t)
	 (#t #\x1F1E6 #t #\x003A #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x003A #t)
	 (#t #\x1F1E6 #t #\x002C #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x002C #t)
	 (#t #\x1F1E6 #t #\x002E #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x002E #t)
	 (#t #\x1F1E6 #t #\x0030 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0030 #t)
	 (#t #\x1F1E6 #t #\x005F #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x005F #t)
	 (#t #\x1F1E6 #f #\x1F1E6 #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x1F1E6 #t)
	 (#t #\x1F1E6 #t #\x05D0 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x1F1E6 #t #\x0022 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0022 #t)
	 (#t #\x1F1E6 #t #\x0027 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0027 #t)
	 (#t #\x1F1E6 #t #\x261D #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x261D #t)
	 (#t #\x1F1E6 #t #\x1F3FB #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1F1E6 #t #\x2764 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F1E6 #t #\x1F466 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F1E6 #f #\x00AD #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x00AD #t)
	 (#t #\x1F1E6 #f #\x0300 #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F1E6 #f #\x200D #t)
	 (#t #\x1F1E6 #f #\x0308 #f #\x200D #t)
	 (#t #\x1F1E6 #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F1E6 #t #\x0061 #t #\x003A #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x1F1E6 #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F1E6 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F1E6 #t #\x0061 #t #\x002C #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x1F1E6 #t #\x0031 #t #\x003A #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x1F1E6 #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F1E6 #t #\x0031 #t #\x002C #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x1F1E6 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F1E6 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x05D0 #t #\x0001 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x0001 #t)
	 (#t #\x05D0 #t #\x000D #t)
	 (#t #\x05D0 #f #\x0308 #t #\x000D #t)
	 (#t #\x05D0 #t #\x000A #t)
	 (#t #\x05D0 #f #\x0308 #t #\x000A #t)
	 (#t #\x05D0 #t #\x000B #t)
	 (#t #\x05D0 #f #\x0308 #t #\x000B #t)
	 (#t #\x05D0 #t #\x3031 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x3031 #t)
	 (#t #\x05D0 #f #\x0041 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0041 #t)
	 (#t #\x05D0 #t #\x003A #t)
	 (#t #\x05D0 #f #\x0308 #t #\x003A #t)
	 (#t #\x05D0 #t #\x002C #t)
	 (#t #\x05D0 #f #\x0308 #t #\x002C #t)
	 (#t #\x05D0 #t #\x002E #t)
	 (#t #\x05D0 #f #\x0308 #t #\x002E #t)
	 (#t #\x05D0 #f #\x0030 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0030 #t)
	 (#t #\x05D0 #f #\x005F #t)
	 (#t #\x05D0 #f #\x0308 #f #\x005F #t)
	 (#t #\x05D0 #t #\x1F1E6 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x05D0 #f #\x05D0 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x05D0 #t #\x0022 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x0022 #t)
	 (#t #\x05D0 #f #\x0027 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0027 #t)
	 (#t #\x05D0 #t #\x261D #t)
	 (#t #\x05D0 #f #\x0308 #t #\x261D #t)
	 (#t #\x05D0 #t #\x1F3FB #t)
	 (#t #\x05D0 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x05D0 #t #\x2764 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x2764 #t)
	 (#t #\x05D0 #t #\x1F466 #t)
	 (#t #\x05D0 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x05D0 #f #\x00AD #t)
	 (#t #\x05D0 #f #\x0308 #f #\x00AD #t)
	 (#t #\x05D0 #f #\x0300 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0300 #t)
	 (#t #\x05D0 #f #\x200D #t)
	 (#t #\x05D0 #f #\x0308 #f #\x200D #t)
	 (#t #\x05D0 #f #\x0061 #f #\x2060 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x05D0 #f #\x0061 #t #\x003A #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x05D0 #f #\x0061 #t #\x0027 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x05D0 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x05D0 #f #\x0061 #t #\x002C #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x05D0 #f #\x0031 #t #\x003A #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x05D0 #f #\x0031 #t #\x0027 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x05D0 #f #\x0031 #t #\x002C #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x05D0 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x05D0 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0022 #t #\x0001 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0022 #t #\x000D #t)
	 (#t #\x0022 #f #\x0308 #t #\x000D #t)
	 (#t #\x0022 #t #\x000A #t)
	 (#t #\x0022 #f #\x0308 #t #\x000A #t)
	 (#t #\x0022 #t #\x000B #t)
	 (#t #\x0022 #f #\x0308 #t #\x000B #t)
	 (#t #\x0022 #t #\x3031 #t)
	 (#t #\x0022 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0022 #t #\x0041 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0022 #t #\x003A #t)
	 (#t #\x0022 #f #\x0308 #t #\x003A #t)
	 (#t #\x0022 #t #\x002C #t)
	 (#t #\x0022 #f #\x0308 #t #\x002C #t)
	 (#t #\x0022 #t #\x002E #t)
	 (#t #\x0022 #f #\x0308 #t #\x002E #t)
	 (#t #\x0022 #t #\x0030 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0022 #t #\x005F #t)
	 (#t #\x0022 #f #\x0308 #t #\x005F #t)
	 (#t #\x0022 #t #\x1F1E6 #t)
	 (#t #\x0022 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0022 #t #\x05D0 #t)
	 (#t #\x0022 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0022 #t #\x0022 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0022 #t #\x0027 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0022 #t #\x261D #t)
	 (#t #\x0022 #f #\x0308 #t #\x261D #t)
	 (#t #\x0022 #t #\x1F3FB #t)
	 (#t #\x0022 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0022 #t #\x2764 #t)
	 (#t #\x0022 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0022 #t #\x1F466 #t)
	 (#t #\x0022 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0022 #f #\x00AD #t)
	 (#t #\x0022 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0022 #f #\x0300 #t)
	 (#t #\x0022 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0022 #f #\x200D #t)
	 (#t #\x0022 #f #\x0308 #f #\x200D #t)
	 (#t #\x0022 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0022 #t #\x0061 #t #\x003A #t)
	 (#t #\x0022 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0022 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0022 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0022 #t #\x0061 #t #\x002C #t)
	 (#t #\x0022 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0022 #t #\x0031 #t #\x003A #t)
	 (#t #\x0022 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0022 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0022 #t #\x0031 #t #\x002C #t)
	 (#t #\x0022 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0022 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0022 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0027 #t #\x0001 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0027 #t #\x000D #t)
	 (#t #\x0027 #f #\x0308 #t #\x000D #t)
	 (#t #\x0027 #t #\x000A #t)
	 (#t #\x0027 #f #\x0308 #t #\x000A #t)
	 (#t #\x0027 #t #\x000B #t)
	 (#t #\x0027 #f #\x0308 #t #\x000B #t)
	 (#t #\x0027 #t #\x3031 #t)
	 (#t #\x0027 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0027 #t #\x0041 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0027 #t #\x003A #t)
	 (#t #\x0027 #f #\x0308 #t #\x003A #t)
	 (#t #\x0027 #t #\x002C #t)
	 (#t #\x0027 #f #\x0308 #t #\x002C #t)
	 (#t #\x0027 #t #\x002E #t)
	 (#t #\x0027 #f #\x0308 #t #\x002E #t)
	 (#t #\x0027 #t #\x0030 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0027 #t #\x005F #t)
	 (#t #\x0027 #f #\x0308 #t #\x005F #t)
	 (#t #\x0027 #t #\x1F1E6 #t)
	 (#t #\x0027 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0027 #t #\x05D0 #t)
	 (#t #\x0027 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0027 #t #\x0022 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0027 #t #\x0027 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0027 #t #\x261D #t)
	 (#t #\x0027 #f #\x0308 #t #\x261D #t)
	 (#t #\x0027 #t #\x1F3FB #t)
	 (#t #\x0027 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0027 #t #\x2764 #t)
	 (#t #\x0027 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0027 #t #\x1F466 #t)
	 (#t #\x0027 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0027 #f #\x00AD #t)
	 (#t #\x0027 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0027 #f #\x0300 #t)
	 (#t #\x0027 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0027 #f #\x200D #t)
	 (#t #\x0027 #f #\x0308 #f #\x200D #t)
	 (#t #\x0027 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0027 #t #\x0061 #t #\x003A #t)
	 (#t #\x0027 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0027 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0027 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0027 #t #\x0061 #t #\x002C #t)
	 (#t #\x0027 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0027 #t #\x0031 #t #\x003A #t)
	 (#t #\x0027 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0027 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0027 #t #\x0031 #t #\x002C #t)
	 (#t #\x0027 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0027 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0027 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x261D #t #\x0001 #t)
	 (#t #\x261D #f #\x0308 #t #\x0001 #t)
	 (#t #\x261D #t #\x000D #t)
	 (#t #\x261D #f #\x0308 #t #\x000D #t)
	 (#t #\x261D #t #\x000A #t)
	 (#t #\x261D #f #\x0308 #t #\x000A #t)
	 (#t #\x261D #t #\x000B #t)
	 (#t #\x261D #f #\x0308 #t #\x000B #t)
	 (#t #\x261D #t #\x3031 #t)
	 (#t #\x261D #f #\x0308 #t #\x3031 #t)
	 (#t #\x261D #t #\x0041 #t)
	 (#t #\x261D #f #\x0308 #t #\x0041 #t)
	 (#t #\x261D #t #\x003A #t)
	 (#t #\x261D #f #\x0308 #t #\x003A #t)
	 (#t #\x261D #t #\x002C #t)
	 (#t #\x261D #f #\x0308 #t #\x002C #t)
	 (#t #\x261D #t #\x002E #t)
	 (#t #\x261D #f #\x0308 #t #\x002E #t)
	 (#t #\x261D #t #\x0030 #t)
	 (#t #\x261D #f #\x0308 #t #\x0030 #t)
	 (#t #\x261D #t #\x005F #t)
	 (#t #\x261D #f #\x0308 #t #\x005F #t)
	 (#t #\x261D #t #\x1F1E6 #t)
	 (#t #\x261D #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x261D #t #\x05D0 #t)
	 (#t #\x261D #f #\x0308 #t #\x05D0 #t)
	 (#t #\x261D #t #\x0022 #t)
	 (#t #\x261D #f #\x0308 #t #\x0022 #t)
	 (#t #\x261D #t #\x0027 #t)
	 (#t #\x261D #f #\x0308 #t #\x0027 #t)
	 (#t #\x261D #t #\x261D #t)
	 (#t #\x261D #f #\x0308 #t #\x261D #t)
	 (#t #\x261D #f #\x1F3FB #t)
	 (#t #\x261D #f #\x0308 #f #\x1F3FB #t)
	 (#t #\x261D #t #\x2764 #t)
	 (#t #\x261D #f #\x0308 #t #\x2764 #t)
	 (#t #\x261D #t #\x1F466 #t)
	 (#t #\x261D #f #\x0308 #t #\x1F466 #t)
	 (#t #\x261D #f #\x00AD #t)
	 (#t #\x261D #f #\x0308 #f #\x00AD #t)
	 (#t #\x261D #f #\x0300 #t)
	 (#t #\x261D #f #\x0308 #f #\x0300 #t)
	 (#t #\x261D #f #\x200D #t)
	 (#t #\x261D #f #\x0308 #f #\x200D #t)
	 (#t #\x261D #t #\x0061 #f #\x2060 #t)
	 (#t #\x261D #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x261D #t #\x0061 #t #\x003A #t)
	 (#t #\x261D #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x261D #t #\x0061 #t #\x0027 #t)
	 (#t #\x261D #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x261D #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x261D #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x261D #t #\x0061 #t #\x002C #t)
	 (#t #\x261D #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x261D #t #\x0031 #t #\x003A #t)
	 (#t #\x261D #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x261D #t #\x0031 #t #\x0027 #t)
	 (#t #\x261D #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x261D #t #\x0031 #t #\x002C #t)
	 (#t #\x261D #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x261D #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x261D #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F3FB #t #\x0001 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F3FB #t #\x000D #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x000D #t)
	 (#t #\x1F3FB #t #\x000A #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x000A #t)
	 (#t #\x1F3FB #t #\x000B #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x000B #t)
	 (#t #\x1F3FB #t #\x3031 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x3031 #t)
	 (#t #\x1F3FB #t #\x0041 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0041 #t)
	 (#t #\x1F3FB #t #\x003A #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x003A #t)
	 (#t #\x1F3FB #t #\x002C #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x002C #t)
	 (#t #\x1F3FB #t #\x002E #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x002E #t)
	 (#t #\x1F3FB #t #\x0030 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0030 #t)
	 (#t #\x1F3FB #t #\x005F #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x005F #t)
	 (#t #\x1F3FB #t #\x1F1E6 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1F3FB #t #\x05D0 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x05D0 #t)
	 (#t #\x1F3FB #t #\x0022 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0022 #t)
	 (#t #\x1F3FB #t #\x0027 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0027 #t)
	 (#t #\x1F3FB #t #\x261D #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x261D #t)
	 (#t #\x1F3FB #t #\x1F3FB #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x1F3FB #t #\x2764 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F3FB #t #\x1F466 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F3FB #f #\x00AD #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x00AD #t)
	 (#t #\x1F3FB #f #\x0300 #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F3FB #f #\x200D #t)
	 (#t #\x1F3FB #f #\x0308 #f #\x200D #t)
	 (#t #\x1F3FB #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F3FB #t #\x0061 #t #\x003A #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x1F3FB #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F3FB #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F3FB #t #\x0061 #t #\x002C #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x1F3FB #t #\x0031 #t #\x003A #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x1F3FB #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F3FB #t #\x0031 #t #\x002C #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x1F3FB #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F3FB #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x2764 #t #\x0001 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0001 #t)
	 (#t #\x2764 #t #\x000D #t)
	 (#t #\x2764 #f #\x0308 #t #\x000D #t)
	 (#t #\x2764 #t #\x000A #t)
	 (#t #\x2764 #f #\x0308 #t #\x000A #t)
	 (#t #\x2764 #t #\x000B #t)
	 (#t #\x2764 #f #\x0308 #t #\x000B #t)
	 (#t #\x2764 #t #\x3031 #t)
	 (#t #\x2764 #f #\x0308 #t #\x3031 #t)
	 (#t #\x2764 #t #\x0041 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0041 #t)
	 (#t #\x2764 #t #\x003A #t)
	 (#t #\x2764 #f #\x0308 #t #\x003A #t)
	 (#t #\x2764 #t #\x002C #t)
	 (#t #\x2764 #f #\x0308 #t #\x002C #t)
	 (#t #\x2764 #t #\x002E #t)
	 (#t #\x2764 #f #\x0308 #t #\x002E #t)
	 (#t #\x2764 #t #\x0030 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0030 #t)
	 (#t #\x2764 #t #\x005F #t)
	 (#t #\x2764 #f #\x0308 #t #\x005F #t)
	 (#t #\x2764 #t #\x1F1E6 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x2764 #t #\x05D0 #t)
	 (#t #\x2764 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x2764 #t #\x0022 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0022 #t)
	 (#t #\x2764 #t #\x0027 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0027 #t)
	 (#t #\x2764 #t #\x261D #t)
	 (#t #\x2764 #f #\x0308 #t #\x261D #t)
	 (#t #\x2764 #t #\x1F3FB #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x2764 #t #\x2764 #t)
	 (#t #\x2764 #f #\x0308 #t #\x2764 #t)
	 (#t #\x2764 #t #\x1F466 #t)
	 (#t #\x2764 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x2764 #f #\x00AD #t)
	 (#t #\x2764 #f #\x0308 #f #\x00AD #t)
	 (#t #\x2764 #f #\x0300 #t)
	 (#t #\x2764 #f #\x0308 #f #\x0300 #t)
	 (#t #\x2764 #f #\x200D #t)
	 (#t #\x2764 #f #\x0308 #f #\x200D #t)
	 (#t #\x2764 #t #\x0061 #f #\x2060 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x2764 #t #\x0061 #t #\x003A #t)
	 (#t #\x2764 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x2764 #t #\x0061 #t #\x0027 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x2764 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x2764 #t #\x0061 #t #\x002C #t)
	 (#t #\x2764 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x2764 #t #\x0031 #t #\x003A #t)
	 (#t #\x2764 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x2764 #t #\x0031 #t #\x0027 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x2764 #t #\x0031 #t #\x002C #t)
	 (#t #\x2764 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x2764 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x2764 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F466 #t #\x0001 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0001 #t)
	 (#t #\x1F466 #t #\x000D #t)
	 (#t #\x1F466 #f #\x0308 #t #\x000D #t)
	 (#t #\x1F466 #t #\x000A #t)
	 (#t #\x1F466 #f #\x0308 #t #\x000A #t)
	 (#t #\x1F466 #t #\x000B #t)
	 (#t #\x1F466 #f #\x0308 #t #\x000B #t)
	 (#t #\x1F466 #t #\x3031 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x3031 #t)
	 (#t #\x1F466 #t #\x0041 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0041 #t)
	 (#t #\x1F466 #t #\x003A #t)
	 (#t #\x1F466 #f #\x0308 #t #\x003A #t)
	 (#t #\x1F466 #t #\x002C #t)
	 (#t #\x1F466 #f #\x0308 #t #\x002C #t)
	 (#t #\x1F466 #t #\x002E #t)
	 (#t #\x1F466 #f #\x0308 #t #\x002E #t)
	 (#t #\x1F466 #t #\x0030 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0030 #t)
	 (#t #\x1F466 #t #\x005F #t)
	 (#t #\x1F466 #f #\x0308 #t #\x005F #t)
	 (#t #\x1F466 #t #\x1F1E6 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x1F466 #t #\x05D0 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x1F466 #t #\x0022 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0022 #t)
	 (#t #\x1F466 #t #\x0027 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0027 #t)
	 (#t #\x1F466 #t #\x261D #t)
	 (#t #\x1F466 #f #\x0308 #t #\x261D #t)
	 (#t #\x1F466 #f #\x1F3FB #t)
	 (#t #\x1F466 #f #\x0308 #f #\x1F3FB #t)
	 (#t #\x1F466 #t #\x2764 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x2764 #t)
	 (#t #\x1F466 #t #\x1F466 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x1F466 #f #\x00AD #t)
	 (#t #\x1F466 #f #\x0308 #f #\x00AD #t)
	 (#t #\x1F466 #f #\x0300 #t)
	 (#t #\x1F466 #f #\x0308 #f #\x0300 #t)
	 (#t #\x1F466 #f #\x200D #t)
	 (#t #\x1F466 #f #\x0308 #f #\x200D #t)
	 (#t #\x1F466 #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x1F466 #t #\x0061 #t #\x003A #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x1F466 #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x1F466 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x1F466 #t #\x0061 #t #\x002C #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x1F466 #t #\x0031 #t #\x003A #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x1F466 #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x1F466 #t #\x0031 #t #\x002C #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x1F466 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x1F466 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x00AD #t #\x0001 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0001 #t)
	 (#t #\x00AD #t #\x000D #t)
	 (#t #\x00AD #f #\x0308 #t #\x000D #t)
	 (#t #\x00AD #t #\x000A #t)
	 (#t #\x00AD #f #\x0308 #t #\x000A #t)
	 (#t #\x00AD #t #\x000B #t)
	 (#t #\x00AD #f #\x0308 #t #\x000B #t)
	 (#t #\x00AD #t #\x3031 #t)
	 (#t #\x00AD #f #\x0308 #t #\x3031 #t)
	 (#t #\x00AD #t #\x0041 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0041 #t)
	 (#t #\x00AD #t #\x003A #t)
	 (#t #\x00AD #f #\x0308 #t #\x003A #t)
	 (#t #\x00AD #t #\x002C #t)
	 (#t #\x00AD #f #\x0308 #t #\x002C #t)
	 (#t #\x00AD #t #\x002E #t)
	 (#t #\x00AD #f #\x0308 #t #\x002E #t)
	 (#t #\x00AD #t #\x0030 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0030 #t)
	 (#t #\x00AD #t #\x005F #t)
	 (#t #\x00AD #f #\x0308 #t #\x005F #t)
	 (#t #\x00AD #t #\x1F1E6 #t)
	 (#t #\x00AD #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x00AD #t #\x05D0 #t)
	 (#t #\x00AD #f #\x0308 #t #\x05D0 #t)
	 (#t #\x00AD #t #\x0022 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0022 #t)
	 (#t #\x00AD #t #\x0027 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0027 #t)
	 (#t #\x00AD #t #\x261D #t)
	 (#t #\x00AD #f #\x0308 #t #\x261D #t)
	 (#t #\x00AD #t #\x1F3FB #t)
	 (#t #\x00AD #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x00AD #t #\x2764 #t)
	 (#t #\x00AD #f #\x0308 #t #\x2764 #t)
	 (#t #\x00AD #t #\x1F466 #t)
	 (#t #\x00AD #f #\x0308 #t #\x1F466 #t)
	 (#t #\x00AD #f #\x00AD #t)
	 (#t #\x00AD #f #\x0308 #f #\x00AD #t)
	 (#t #\x00AD #f #\x0300 #t)
	 (#t #\x00AD #f #\x0308 #f #\x0300 #t)
	 (#t #\x00AD #f #\x200D #t)
	 (#t #\x00AD #f #\x0308 #f #\x200D #t)
	 (#t #\x00AD #t #\x0061 #f #\x2060 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x00AD #t #\x0061 #t #\x003A #t)
	 (#t #\x00AD #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x00AD #t #\x0061 #t #\x0027 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x00AD #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x00AD #t #\x0061 #t #\x002C #t)
	 (#t #\x00AD #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x00AD #t #\x0031 #t #\x003A #t)
	 (#t #\x00AD #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x00AD #t #\x0031 #t #\x0027 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x00AD #t #\x0031 #t #\x002C #t)
	 (#t #\x00AD #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x00AD #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x00AD #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0300 #t #\x0001 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0300 #t #\x000D #t)
	 (#t #\x0300 #f #\x0308 #t #\x000D #t)
	 (#t #\x0300 #t #\x000A #t)
	 (#t #\x0300 #f #\x0308 #t #\x000A #t)
	 (#t #\x0300 #t #\x000B #t)
	 (#t #\x0300 #f #\x0308 #t #\x000B #t)
	 (#t #\x0300 #t #\x3031 #t)
	 (#t #\x0300 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0300 #t #\x0041 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0300 #t #\x003A #t)
	 (#t #\x0300 #f #\x0308 #t #\x003A #t)
	 (#t #\x0300 #t #\x002C #t)
	 (#t #\x0300 #f #\x0308 #t #\x002C #t)
	 (#t #\x0300 #t #\x002E #t)
	 (#t #\x0300 #f #\x0308 #t #\x002E #t)
	 (#t #\x0300 #t #\x0030 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0300 #t #\x005F #t)
	 (#t #\x0300 #f #\x0308 #t #\x005F #t)
	 (#t #\x0300 #t #\x1F1E6 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0300 #t #\x05D0 #t)
	 (#t #\x0300 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0300 #t #\x0022 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0300 #t #\x0027 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0300 #t #\x261D #t)
	 (#t #\x0300 #f #\x0308 #t #\x261D #t)
	 (#t #\x0300 #t #\x1F3FB #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0300 #t #\x2764 #t)
	 (#t #\x0300 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0300 #t #\x1F466 #t)
	 (#t #\x0300 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0300 #f #\x00AD #t)
	 (#t #\x0300 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0300 #f #\x0300 #t)
	 (#t #\x0300 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0300 #f #\x200D #t)
	 (#t #\x0300 #f #\x0308 #f #\x200D #t)
	 (#t #\x0300 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0300 #t #\x0061 #t #\x003A #t)
	 (#t #\x0300 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0300 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0300 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0300 #t #\x0061 #t #\x002C #t)
	 (#t #\x0300 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0300 #t #\x0031 #t #\x003A #t)
	 (#t #\x0300 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0300 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0300 #t #\x0031 #t #\x002C #t)
	 (#t #\x0300 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0300 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0300 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x200D #t #\x0001 #t)
	 (#t #\x200D #f #\x0308 #t #\x0001 #t)
	 (#t #\x200D #t #\x000D #t)
	 (#t #\x200D #f #\x0308 #t #\x000D #t)
	 (#t #\x200D #t #\x000A #t)
	 (#t #\x200D #f #\x0308 #t #\x000A #t)
	 (#t #\x200D #t #\x000B #t)
	 (#t #\x200D #f #\x0308 #t #\x000B #t)
	 (#t #\x200D #t #\x3031 #t)
	 (#t #\x200D #f #\x0308 #t #\x3031 #t)
	 (#t #\x200D #t #\x0041 #t)
	 (#t #\x200D #f #\x0308 #t #\x0041 #t)
	 (#t #\x200D #t #\x003A #t)
	 (#t #\x200D #f #\x0308 #t #\x003A #t)
	 (#t #\x200D #t #\x002C #t)
	 (#t #\x200D #f #\x0308 #t #\x002C #t)
	 (#t #\x200D #t #\x002E #t)
	 (#t #\x200D #f #\x0308 #t #\x002E #t)
	 (#t #\x200D #t #\x0030 #t)
	 (#t #\x200D #f #\x0308 #t #\x0030 #t)
	 (#t #\x200D #t #\x005F #t)
	 (#t #\x200D #f #\x0308 #t #\x005F #t)
	 (#t #\x200D #t #\x1F1E6 #t)
	 (#t #\x200D #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x200D #t #\x05D0 #t)
	 (#t #\x200D #f #\x0308 #t #\x05D0 #t)
	 (#t #\x200D #t #\x0022 #t)
	 (#t #\x200D #f #\x0308 #t #\x0022 #t)
	 (#t #\x200D #t #\x0027 #t)
	 (#t #\x200D #f #\x0308 #t #\x0027 #t)
	 (#t #\x200D #t #\x261D #t)
	 (#t #\x200D #f #\x0308 #t #\x261D #t)
	 (#t #\x200D #t #\x1F3FB #t)
	 (#t #\x200D #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x200D #f #\x2764 #t)
	 (#t #\x200D #f #\x0308 #t #\x2764 #t)
	 (#t #\x200D #f #\x1F466 #t)
	 (#t #\x200D #f #\x0308 #t #\x1F466 #t)
	 (#t #\x200D #f #\x00AD #t)
	 (#t #\x200D #f #\x0308 #f #\x00AD #t)
	 (#t #\x200D #f #\x0300 #t)
	 (#t #\x200D #f #\x0308 #f #\x0300 #t)
	 (#t #\x200D #f #\x200D #t)
	 (#t #\x200D #f #\x0308 #f #\x200D #t)
	 (#t #\x200D #t #\x0061 #f #\x2060 #t)
	 (#t #\x200D #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x200D #t #\x0061 #t #\x003A #t)
	 (#t #\x200D #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x200D #t #\x0061 #t #\x0027 #t)
	 (#t #\x200D #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x200D #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x200D #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x200D #t #\x0061 #t #\x002C #t)
	 (#t #\x200D #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x200D #t #\x0031 #t #\x003A #t)
	 (#t #\x200D #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x200D #t #\x0031 #t #\x0027 #t)
	 (#t #\x200D #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x200D #t #\x0031 #t #\x002C #t)
	 (#t #\x200D #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x200D #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x200D #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #t #\x0001 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0061 #f #\x2060 #t #\x000D #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x000D #t)
	 (#t #\x0061 #f #\x2060 #t #\x000A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x000A #t)
	 (#t #\x0061 #f #\x2060 #t #\x000B #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x000B #t)
	 (#t #\x0061 #f #\x2060 #t #\x3031 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0041 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0041 #t)
	 (#t #\x0061 #f #\x2060 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #t #\x002E #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x002E #t)
	 (#t #\x0061 #f #\x2060 #f #\x0030 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0030 #t)
	 (#t #\x0061 #f #\x2060 #f #\x005F #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x005F #t)
	 (#t #\x0061 #f #\x2060 #t #\x1F1E6 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0061 #f #\x2060 #f #\x05D0 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0061 #f #\x2060 #t #\x0022 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0061 #f #\x2060 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #t #\x261D #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x261D #t)
	 (#t #\x0061 #f #\x2060 #t #\x1F3FB #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0061 #f #\x2060 #t #\x2764 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0061 #f #\x2060 #t #\x1F466 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0061 #f #\x2060 #f #\x00AD #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0061 #f #\x2060 #f #\x0300 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0061 #f #\x2060 #f #\x200D #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x200D #t)
	 (#t #\x0061 #f #\x2060 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #f #\x0031 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0061 #f #\x2060 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0031 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0061 #f #\x2060 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #f #\x2060 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x003A #t #\x0001 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0001 #t)
	 (#t #\x0061 #t #\x003A #t #\x000D #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x000D #t)
	 (#t #\x0061 #t #\x003A #t #\x000A #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x000A #t)
	 (#t #\x0061 #t #\x003A #t #\x000B #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x000B #t)
	 (#t #\x0061 #t #\x003A #t #\x3031 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x3031 #t)
	 (#t #\x0061 #f #\x003A #f #\x0041 #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0041 #t)
	 (#t #\x0061 #t #\x003A #t #\x003A #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x003A #t)
	 (#t #\x0061 #t #\x003A #t #\x002C #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x002C #t)
	 (#t #\x0061 #t #\x003A #t #\x002E #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x002E #t)
	 (#t #\x0061 #t #\x003A #t #\x0030 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0030 #t)
	 (#t #\x0061 #t #\x003A #t #\x005F #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x005F #t)
	 (#t #\x0061 #t #\x003A #t #\x1F1E6 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0061 #f #\x003A #f #\x05D0 #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0061 #t #\x003A #t #\x0022 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0022 #t)
	 (#t #\x0061 #t #\x003A #t #\x0027 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0027 #t)
	 (#t #\x0061 #t #\x003A #t #\x261D #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x261D #t)
	 (#t #\x0061 #t #\x003A #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x003A #t #\x2764 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x2764 #t)
	 (#t #\x0061 #t #\x003A #t #\x1F466 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x003A #f #\x00AD #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #f #\x00AD #t)
	 (#t #\x0061 #t #\x003A #f #\x0300 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #f #\x0300 #t)
	 (#t #\x0061 #t #\x003A #f #\x200D #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #f #\x200D #t)
	 (#t #\x0061 #f #\x003A #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x003A #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x003A #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x003A #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x003A #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #f #\x003A #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #t #\x003A #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x003A #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x003A #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x003A #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x003A #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x0027 #t #\x0001 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0061 #t #\x0027 #t #\x000D #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x000D #t)
	 (#t #\x0061 #t #\x0027 #t #\x000A #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x000A #t)
	 (#t #\x0061 #t #\x0027 #t #\x000B #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x000B #t)
	 (#t #\x0061 #t #\x0027 #t #\x3031 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0041 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0041 #t)
	 (#t #\x0061 #t #\x0027 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #t #\x002E #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x002E #t)
	 (#t #\x0061 #t #\x0027 #t #\x0030 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0061 #t #\x0027 #t #\x005F #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x005F #t)
	 (#t #\x0061 #t #\x0027 #t #\x1F1E6 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0061 #f #\x0027 #f #\x05D0 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0061 #t #\x0027 #t #\x0022 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0061 #t #\x0027 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #t #\x261D #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x261D #t)
	 (#t #\x0061 #t #\x0027 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x0027 #t #\x2764 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0061 #t #\x0027 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x0027 #f #\x00AD #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0061 #t #\x0027 #f #\x0300 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0061 #t #\x0027 #f #\x200D #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #f #\x200D #t)
	 (#t #\x0061 #f #\x0027 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x0027 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #f #\x0027 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x0027 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0001 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x000D #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x000D #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x000A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x000A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x000B #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x000B #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x3031 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0041 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0041 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x002E #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x002E #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0030 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0030 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x005F #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x005F #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x1F1E6 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x05D0 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x05D0 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0022 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x261D #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x261D #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x2764 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x00AD #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0300 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x200D #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #f #\x200D #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x003A #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #f #\x0027 #f #\x2060 #f #\x0308 #f #\x0061 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x0027 #f #\x2060 #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #t #\x0001 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0001 #t)
	 (#t #\x0061 #t #\x002C #t #\x000D #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x000D #t)
	 (#t #\x0061 #t #\x002C #t #\x000A #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x000A #t)
	 (#t #\x0061 #t #\x002C #t #\x000B #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x000B #t)
	 (#t #\x0061 #t #\x002C #t #\x3031 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x3031 #t)
	 (#t #\x0061 #t #\x002C #t #\x0041 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0041 #t)
	 (#t #\x0061 #t #\x002C #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #t #\x002E #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x002E #t)
	 (#t #\x0061 #t #\x002C #t #\x0030 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0030 #t)
	 (#t #\x0061 #t #\x002C #t #\x005F #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x005F #t)
	 (#t #\x0061 #t #\x002C #t #\x1F1E6 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0061 #t #\x002C #t #\x05D0 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0061 #t #\x002C #t #\x0022 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0022 #t)
	 (#t #\x0061 #t #\x002C #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #t #\x261D #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x261D #t)
	 (#t #\x0061 #t #\x002C #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0061 #t #\x002C #t #\x2764 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x2764 #t)
	 (#t #\x0061 #t #\x002C #t #\x1F466 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0061 #t #\x002C #f #\x00AD #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #f #\x00AD #t)
	 (#t #\x0061 #t #\x002C #f #\x0300 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #f #\x0300 #t)
	 (#t #\x0061 #t #\x002C #f #\x200D #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #f #\x200D #t)
	 (#t #\x0061 #t #\x002C #t #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #t #\x0061 #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #t #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #t #\x0061 #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0061 #t #\x002C #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0061 #t #\x002C #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0061 #t #\x002C #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0061 #t #\x002C #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #t #\x0001 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0001 #t)
	 (#t #\x0031 #t #\x003A #t #\x000D #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x000D #t)
	 (#t #\x0031 #t #\x003A #t #\x000A #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x000A #t)
	 (#t #\x0031 #t #\x003A #t #\x000B #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x000B #t)
	 (#t #\x0031 #t #\x003A #t #\x3031 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x3031 #t)
	 (#t #\x0031 #t #\x003A #t #\x0041 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0041 #t)
	 (#t #\x0031 #t #\x003A #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #t #\x002E #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x002E #t)
	 (#t #\x0031 #t #\x003A #t #\x0030 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0030 #t)
	 (#t #\x0031 #t #\x003A #t #\x005F #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x005F #t)
	 (#t #\x0031 #t #\x003A #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x003A #t #\x05D0 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x003A #t #\x0022 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0022 #t)
	 (#t #\x0031 #t #\x003A #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #t #\x261D #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x261D #t)
	 (#t #\x0031 #t #\x003A #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x003A #t #\x2764 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x2764 #t)
	 (#t #\x0031 #t #\x003A #t #\x1F466 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x003A #f #\x00AD #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #f #\x00AD #t)
	 (#t #\x0031 #t #\x003A #f #\x0300 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #f #\x0300 #t)
	 (#t #\x0031 #t #\x003A #f #\x200D #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #f #\x200D #t)
	 (#t #\x0031 #t #\x003A #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #t #\x0031 #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0031 #t #\x003A #t)
	 (#t #\x0031 #t #\x003A #t #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #t #\x003A #t #\x0031 #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0031 #t #\x002C #t)
	 (#t #\x0031 #t #\x003A #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #t #\x003A #f #\x0308 #t #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0001 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0031 #t #\x0027 #t #\x000D #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x000D #t)
	 (#t #\x0031 #t #\x0027 #t #\x000A #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x000A #t)
	 (#t #\x0031 #t #\x0027 #t #\x000B #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x000B #t)
	 (#t #\x0031 #t #\x0027 #t #\x3031 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0041 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0031 #t #\x0027 #t #\x003A #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x003A #t)
	 (#t #\x0031 #t #\x0027 #t #\x002C #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x002C #t)
	 (#t #\x0031 #t #\x0027 #t #\x002E #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x002E #t)
	 (#t #\x0031 #f #\x0027 #f #\x0030 #t)
	 (#t #\x0031 #f #\x0027 #f #\x0308 #f #\x0030 #t)
	 (#t #\x0031 #t #\x0027 #t #\x005F #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x005F #t)
	 (#t #\x0031 #t #\x0027 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x0027 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0022 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0027 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0031 #t #\x0027 #t #\x261D #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x261D #t)
	 (#t #\x0031 #t #\x0027 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x0027 #t #\x2764 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0031 #t #\x0027 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x0027 #f #\x00AD #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0031 #t #\x0027 #f #\x0300 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0031 #t #\x0027 #f #\x200D #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #f #\x200D #t)
	 (#t #\x0031 #t #\x0027 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x0027 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x0027 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #t #\x0027 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #f #\x0027 #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x0027 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x0027 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x0027 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x0027 #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x0027 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x0027 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #f #\x0027 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #t #\x002C #t #\x0001 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0001 #t)
	 (#t #\x0031 #t #\x002C #t #\x000D #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x000D #t)
	 (#t #\x0031 #t #\x002C #t #\x000A #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x000A #t)
	 (#t #\x0031 #t #\x002C #t #\x000B #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x000B #t)
	 (#t #\x0031 #t #\x002C #t #\x3031 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x3031 #t)
	 (#t #\x0031 #t #\x002C #t #\x0041 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0041 #t)
	 (#t #\x0031 #t #\x002C #t #\x003A #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x003A #t)
	 (#t #\x0031 #t #\x002C #t #\x002C #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x002C #t)
	 (#t #\x0031 #t #\x002C #t #\x002E #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x002E #t)
	 (#t #\x0031 #f #\x002C #f #\x0030 #t)
	 (#t #\x0031 #f #\x002C #f #\x0308 #f #\x0030 #t)
	 (#t #\x0031 #t #\x002C #t #\x005F #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x005F #t)
	 (#t #\x0031 #t #\x002C #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x002C #t #\x05D0 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x002C #t #\x0022 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0022 #t)
	 (#t #\x0031 #t #\x002C #t #\x0027 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002C #t #\x261D #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x261D #t)
	 (#t #\x0031 #t #\x002C #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x002C #t #\x2764 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x2764 #t)
	 (#t #\x0031 #t #\x002C #t #\x1F466 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x002C #f #\x00AD #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #f #\x00AD #t)
	 (#t #\x0031 #t #\x002C #f #\x0300 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #f #\x0300 #t)
	 (#t #\x0031 #t #\x002C #f #\x200D #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #f #\x200D #t)
	 (#t #\x0031 #t #\x002C #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002C #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x002C #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002C #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002C #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #t #\x002C #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #f #\x002C #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x002C #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x002C #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x002C #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x002C #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x002C #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x002C #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #f #\x002C #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0001 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0001 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x000D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x000D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x000A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x000A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x000B #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x000B #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x3031 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x3031 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0041 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0041 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x003A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x003A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x002C #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x002C #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x002E #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x002E #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0030 #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0308 #f #\x0030 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x005F #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x005F #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x1F1E6 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x05D0 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0022 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0022 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x261D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x261D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x1F3FB #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x2764 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x2764 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x1F466 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x00AD #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #f #\x00AD #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0300 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #f #\x0300 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x200D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #f #\x200D #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0061 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0061 #t #\x003A #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0061 #t #\x0027 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0061 #t #\x0027 #f #\x2060 #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #t #\x002E #f #\x2060 #f #\x0308 #t #\x0061 #t #\x002C #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0308 #f #\x0031 #t #\x003A #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0308 #f #\x0031 #t #\x0027 #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0308 #f #\x0031 #t #\x002C #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x0031 #f #\x002E #f #\x2060 #f #\x0308 #f #\x0031 #t #\x002E #f #\x2060 #t)
	 (#t #\x000D #f #\x000A #t #\x0061 #t #\x000A #t #\x0308 #t)
	 (#t #\x0061 #f #\x0308 #t)
	 (#t #\x0020 #f #\x200D #t #\x0646 #t)
	 (#t #\x0646 #f #\x200D #t #\x0020 #t)
	 (#t #\x0041 #f #\x0041 #f #\x0041 #t)
	 (#t #\x0041 #f #\x003A #f #\x0041 #t)
	 (#t #\x0041 #t #\x003A #t #\x003A #t #\x0041 #t)
	 (#t #\x05D0 #f #\x0027 #t)
	 (#t #\x05D0 #f #\x0022 #f #\x05D0 #t)
	 (#t #\x0041 #f #\x0030 #f #\x0030 #f #\x0041 #t)
	 (#t #\x0030 #f #\x002C #f #\x0030 #t)
	 (#t #\x0030 #t #\x002C #t #\x002C #t #\x0030 #t)
	 (#t #\x3031 #f #\x3031 #t)
	 (#t #\x0041 #f #\x005F #f #\x0030 #f #\x005F #f #\x3031 #f #\x005F #t)
	 (#t #\x0041 #f #\x005F #f #\x005F #f #\x0041 #t)
	 (#t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #f #\x200D #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x200D #f #\x1F1E7 #t #\x1F1E8 #t #\x0062 #t)
	 (#t #\x0061 #t #\x1F1E6 #f #\x1F1E7 #t #\x1F1E8 #f #\x1F1E9 #t #\x0062 #t)
	 (#t #\x261D #f #\x1F3FB #t #\x261D #t)
	 (#t #\x1F466 #f #\x1F3FB #t)
	 (#t #\x200D #f #\x1F466 #f #\x1F3FB #t)
	 (#t #\x200D #f #\x2764 #t)
	 (#t #\x200D #f #\x1F466 #t)
	 (#t #\x1F466 #t #\x1F466 #t)
	 (#t #\x0061 #f #\x0308 #f #\x200D #f #\x0308 #f #\x0062 #t)
	 (#t #\x0031 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0031 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0031 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0031 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0031 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0031 #t #\x002C #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0031 #t #\x002C #t #\x002C #t #\x0061 #t)
	 (#t #\x0031 #f #\x005F #f #\x0061 #t #\x002C #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x003A #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002E #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x003A #t #\x0031 #t)
	 (#t #\x0061 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x003A #t #\x0061 #t)
	 (#t #\x0061 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x002E #t #\x0031 #t)
	 (#t #\x0061 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x002E #t #\x0061 #t)
	 (#t #\x0061 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x002C #t #\x0031 #t)
	 (#t #\x0061 #t #\x002C #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0031 #t #\x002C #t #\x002C #t #\x0061 #t)
	 (#t #\x0061 #f #\x005F #f #\x0061 #t #\x002C #t #\x002C #t #\x0061 #t))))

(define-test 'string-trim
  (lambda ()
    (define-integrable = assert-string=)
    (= "foo" (string-trim "foo   "))
    (= "foo" (string-trim "   foo"))
    (= "foo" (string-trim "   foo   "))
    (= "foo   " (string-trim-left "   foo   "))
    (= "   foo" (string-trim-right "   foo   "))
    (= "" (string-trim "\"\"" (char-set-invert (char-set #\"))))
    (= "" (string-trim-left "\"\"" (char-set-invert (char-set #\"))))
    (= "" (string-trim-right "\"\"" (char-set-invert (char-set #\"))))
    (= "foo" (string-trim "aaafooaaa" (char-set #\f #\o)))
    (= "fooaaa" (string-trim-left "aaafooaaa" (char-set #\f #\o)))
    (= "aaafoo" (string-trim-right "aaafooaaa" (char-set #\f #\o)))))