/*
 *  Copyright 1994-2019 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include <libswscale/swscale.h>
#include "image_8bits.h"
#include "pnglite.h"


Image8_t *
Image8_new()
{
  Image8_t *p = NULL;

  p = xcalloc(1, sizeof(Image8_t));

  p->id = -1;
  p->buff = Buffer8_new();

  return p;
}


Image8_t *
Image8_new_default()
{
  uint32_t i;
  Image8_t *img = Image8_new();

  for (i = 0; i < BUFFSIZE; i++)
    img->buff->buffer[i] = i;

  return img;
}


void
Image8_delete(Image8_t *p)
{
  xfree(p->name);
  xfree(p->dname);

  if (p->buff)
    Buffer8_delete(p->buff);
  xfree(p);
}


int
Image8_load_PNG(Image8_t *pic, const char *filename)
{
  png_t png;
  int res;

  res = png_open_file_read(&png, filename);
  if (PNG_NO_ERROR == res) {
    size_t size = png.width * png.height * png.bpp;
    u_char *data = xmalloc(size * sizeof(u_char));

    res = png_get_data(&png, data);
    if (PNG_NO_ERROR == res) {
      /* source */
      enum AVPixelFormat srcFormat = AV_PIX_FMT_NONE;
      int srcStride[4] = { 0, 0, 0, 0 };
      const uint8_t *srcSlice[4] = { NULL, NULL, NULL, NULL };
      /* destination */
      const enum AVPixelFormat dstFormat = AV_PIX_FMT_GRAY8;
      const int dstStride[4] = { WIDTH, 0, 0, 0 };
      uint8_t *dstSlice[4] = { NULL, NULL, NULL, NULL };
      struct SwsContext *sws_context = NULL;
      int ret;

      srcSlice[0] = (const uint8_t *)data;
      dstSlice[0] = (uint8_t *)pic->buff->buffer;

      /* set srcFormat from PNG infos */
      switch (png.color_type) {
      case PNG_GREYSCALE:
	srcFormat = AV_PIX_FMT_GRAY8;
	srcStride[0] = png.width;
	break;

      case PNG_TRUECOLOR:
	srcFormat = AV_PIX_FMT_RGB24;
	srcStride[0] = png.width * 3;
	break;

      case PNG_INDEXED:
	/* TODO: unhandled */
	xerror("Image8_load_PNG: PNG_INDEXED not yet implemented\n");
	break;

      case PNG_GREYSCALE_ALPHA:
	/* TODO: unhandled */
	xerror("Image8_load_PNG: PNG_GREYSCALE_ALPHA not yet implemented\n");
	break;

      case PNG_TRUECOLOR_ALPHA:
	srcFormat = AV_PIX_FMT_RGBA;
	srcStride[0] = png.width * 4;
	break;

      default:
	xerror("Image8_load_PNG: unknown PNG color type %d\n", png.color_type);
	break;
      }

      /* create context */
      sws_context = sws_getContext(png.width, png.height, srcFormat,
				   WIDTH, HEIGHT, dstFormat,
#ifdef __NetBSD__
				   SWS_BILINEAR, NULL, NULL, NULL
#else
                                   SWS_FAST_BILINEAR, NULL, NULL, NULL
#endif
				   );
      if (NULL == sws_context)
	xerror("sws_getContext\n");

      /* rescale (size and color space) */
      ret = sws_scale(sws_context,
		      srcSlice,
		      srcStride,
		      0,
		      png.height,
		      dstSlice,
		      dstStride);
      if (ret < 0)
	xerror("sws_scale\n");

      /* free context and data */
      sws_freeContext(sws_context);
      xfree(data);
      /* flip image */
      Buffer8_flip_v(pic->buff);
    } else {
      static int verbose1 = 1;

      fprintf(stderr, "[!] png_get_data: %s (%s)\n", png_error_string(res), filename);
      if (verbose1) {
	fprintf(stderr, "[!] if this is an indexed PNG, convert it to non-indexed\n");
	fprintf(stderr, "[!] also, remove any comments from the PNG\n");
	verbose1 = 0;
      }
      xfree(data);
      png_close_file(&png);

      return -1;
    }
  } else {
    static int verbose2 = 1;

    fprintf(stderr, "[!] png_open_file_read: %s (%s)\n", png_error_string(res), filename);
    if (verbose2) {
      fprintf(stderr, "[!] Please remove any comments from this PNG file\n");
      verbose2 = 0;
    }
    if (PNG_FILE_ERROR != res)
      png_close_file(&png);

    return -1;
  }

  png_close_file(&png);

  return 0;
}


int
Image8_load(Image8_t *pic, const uint32_t id, const char *dir, const char *filename)
{
  char *file = g_strdup_printf("%s/%s", dir, filename);

  if (Image8_load_PNG(pic, file) != 0) {
    g_free(file);
    return (-1);
  }
  g_free(file);

  xfree(pic->name);
  pic->name = strdup(filename);
  xfree(pic->dname);
  pic->dname = strdup(pic->name);

  if ((file = strchr(pic->dname, '.')) != NULL)
    *file = '\0'; /* spr0tch */

  pic->id = id;

  return (0);
}


void
Image8_copy(const Image8_t *from, Image8_t *to)
{
  assert(from != NULL);
  assert(to != NULL);

  assert(from->name != NULL);

  xfree(to->name);

  to->name = strdup(from->name);

  to->id = from->id;

  Buffer8_copy(from->buff, to->buff);
}
