/*
 * Copyright (C) 2003-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include "arch_dma.c"
#include "arch_pic.c"
#include "arch_pit.c"
#include "arch_ppi.c"
#include "arch_rtc.c"

#endif
#ifdef STATE

/* FIXME */
uint8_t state_inta;

/* FIXME */
uint8_t dma_page[16];   /* 0: reserved */
			/* 1: ctrl 0, chan 2 */
			/* 2: ctrl 0, chan 3 */
			/* 3: ctrl 0, chan 1 */
			/* 4: reserved */
			/* 5: reserved */
			/* 6: reserved */
			/* 7: ctrl 0, chan 0 */
			/* 8: reserved */
			/* 9: ctrl 1, chan 2 */
			/* A: ctrl 1, chan 3 */
			/* B: ctrl 1, chan 1 */
			/* C: reserved */
			/* D: reserved */
			/* E: reserved */
			/* F: ctrl 1, chan 0 */

/* First DMA Controller */
#define NAME            dma0
#define NAME_(x)        dma0_ ## x
#define SNAME           "dma0"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

/* Second DMA Controller */
#define NAME            dma1
#define NAME_(x)        dma1_ ## x
#define SNAME           "dma1"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

/* First PIC (master) */
#define NAME            pic0
#define NAME_(x)        pic0_ ## x
#define SNAME           "pic0"
#include "arch_pic.c"
#undef SNAME
#undef NAME_
#undef NAME

/* Second PIC (slave) */
#define NAME            pic1
#define NAME_(x)        pic1_ ## x
#define SNAME           "pic1"
#include "arch_pic.c"
#undef SNAME
#undef NAME_
#undef NAME

/* PIT */
#define NAME            pit
#define NAME_(x)        pit_ ## x
#define SNAME           "pit"
#include "arch_pit.c"
#undef SNAME
#undef NAME_
#undef NAME

/* PPI */
#define NAME            ppi
#define NAME_(x)        ppi_ ## x
#define SNAME           "ppi"
#include "arch_ppi.c"
#undef SNAME
#undef NAME_
#undef NAME

/* RTC */
#define NAME            rtc
#define NAME_(x)        rtc_ ## x
#define SNAME           "rtc"
#include "arch_rtc.c"
#undef SNAME
#undef NAME_
#undef NAME

#endif /* STATE */

/* --------------------------------------------------------------------- */

#ifdef BEHAVIOR

/*forward*/ static void
pic0_int2_set(struct cpssp *cpssp, unsigned int value);

static inline void
dma0_ack_out(
	struct cpssp *cpssp,
	uint16_t offset,
	unsigned int chan,
	unsigned int tc
)
{
	uint8_t page;
	unsigned long pa;

	assert(/* 0 <= chan && */ chan < 4);

	page = cpssp->dma_page[chanreg[chan + 0]];

	pa = (page << 16) | offset;

	legacy_dma_ack_outb(cpssp, pa, chan, tc);
}

static inline void
dma0_ack_in(
	struct cpssp *cpssp,
        uint16_t offset,
        unsigned int chan,
        unsigned int tc
)
{
        uint8_t page;
        unsigned long pa;

        assert(/* 0 <= chan && */ chan < 4);

        page = cpssp->dma_page[chanreg[chan + 0]];

        pa = (page << 16) | offset;

	legacy_dma_ack_inb(cpssp, pa, chan, tc);
}

static inline void
dma0_ack_verify(
	struct cpssp *cpssp,
        unsigned int chan,
        unsigned int tc
)
{
        assert(/* 0 <= chan && */ chan < 4);

	legacy_dma_ack_verifyb(cpssp, chan, tc);
}

static inline void
dma1_ack_out(
        struct cpssp *cpssp,
        uint16_t offset,
        unsigned int chan,
        unsigned int tc
)
{
        uint8_t page;
        unsigned long pa;

        assert(/* 0 <= chan && */ chan < 4);

        page = cpssp->dma_page[chanreg[chan + 4]];

        pa = (page << 16) | (offset << 1);

	legacy_dma_ack_outw(cpssp, pa, chan, tc);
}

static inline void
dma1_ack_in(
        struct cpssp *cpssp,
        uint16_t offset,
        unsigned int chan,
        unsigned int tc
)
{
        uint8_t page;
        unsigned long pa;

        assert(/* 0 <= chan && */ chan < 4);

        page = cpssp->dma_page[chanreg[chan + 4]];

        pa = (page << 16) | (offset << 1);

	legacy_dma_ack_inw(cpssp, pa, chan, tc);
}

static inline void
dma1_ack_verify(
        struct cpssp *cpssp,
        unsigned int chan,
        unsigned int tc
)
{
	legacy_dma_ack_verifyw(cpssp, chan, tc);
}

static void
pic0_cas_out_set(struct cpssp *cpssp, unsigned int value)
{
        pic1_cas_in_set(cpssp, value);
}

static inline void
pic1_irq_out_set(struct cpssp *cpssp, unsigned int value)
{
        pic0_int2_set(cpssp, value);
}

static void
pic1_cas_out_set(struct cpssp *cpssp, unsigned int value)
{
        fixme();
}

/* First DMA Controller */
#define NAME            dma0
#define NAME_(x)        dma0_ ## x
#define SNAME           "dma0"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

/* Second DMA Controller */
#define NAME            dma1
#define NAME_(x)        dma1_ ## x
#define SNAME           "dma1"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

/* First PIC (master) */
#define MASTER          1
#define ELCR_MASK       0xf8    /* Can't set bits 0-2. */
#define NAME            pic0
#define NAME_(x)        pic0_ ## x
#define SNAME           "pic0"
#include "arch_pic.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef ELCR_MASK
#undef MASTER

/* Second PIC (slave) */
#define MASTER          0
#define ELCR_MASK       0xde    /* Can't set bits 0 and 5. */
#define NAME            pic1
#define NAME_(x)        pic1_ ## x
#define SNAME           "pic1"
#include "arch_pic.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef ELCR_MASK
#undef MASTER

/* PIT */
#define NAME            pit
#define NAME_(x)        pit_ ## x
#define SNAME           "pit"
#include "arch_pit.c"
#undef SNAME
#undef NAME_
#undef NAME

/* PPI */
#define NAME            ppi
#define NAME_(x)        ppi_ ## x
#define SNAME           "ppi"
#include "arch_ppi.c"
#undef SNAME
#undef NAME_
#undef NAME

/* RTC */
#define NAME            rtc
#define NAME_(x)        rtc_ ## x
#define SNAME           "rtc"
#include "arch_rtc.c"
#undef SNAME
#undef NAME_
#undef NAME

static void
legacy_int0_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int0_set(cpssp, val);
}

static void
legacy_int1_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int1_set(cpssp, val);
}

static void
legacy_int3_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int3_set(cpssp, val);
}

static void
legacy_int4_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int4_set(cpssp, val);
}

static void
legacy_int5_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int5_set(cpssp, val);
}

static void
legacy_int6_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int6_set(cpssp, val);
}

static void
legacy_int7_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic0_int7_set(cpssp, val);
}

static void
legacy_int8_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int0_set(cpssp, val);
}

static void
legacy_int9_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int1_set(cpssp, val);
}

static void
legacy_int10_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int2_set(cpssp, val);
}

static void
legacy_int11_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int3_set(cpssp, val);
}

static void
legacy_int12_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int4_set(cpssp, val);
}

static void
legacy_int13_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int5_set(cpssp, val);
}

static void
legacy_int14_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int6_set(cpssp, val);
}

static void
legacy_int15_set(void *_cpssp, unsigned int val)
{
        struct cpssp *cpssp = _cpssp;

        pic1_int7_set(cpssp, val);
}

static int
legacy_inta_addr(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	pic0_inta_begin(cpssp, &cpssp->state_inta);
        pic1_inta_begin(cpssp, &cpssp->state_inta);

	return 0;
}

static int
legacy_inta_data(void *_cpssp, uint8_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	pic1_inta_end(cpssp);
        pic0_inta_end(cpssp);

        *valp = cpssp->state_inta;

	return 0;
}

static int
legacy_inb(struct cpssp *cpssp, uint32_t port, uint8_t *valp)
{
	switch (port) {
        case 0x0000 ... 0x001f:
                dma0_inb(cpssp, valp, port & 0xf);
                return 0;

	case 0x0020: case 0x0021: case 0x0024: case 0x0025:
        case 0x0028: case 0x0029: case 0x002c: case 0x002d:
        case 0x0030: case 0x0031: case 0x0034: case 0x0035:
        case 0x0038: case 0x0039: case 0x003c: case 0x003d:
                pic0_inb(cpssp, valp, port & 0x1);
                return 0;

        case 0x0040 ... 0x0043:
        case 0x0050 ... 0x0053:
                /* PIT 0 */
                pit_inb(cpssp, valp, port & 0x3);
                return 0;

	case 0x0061: case 0x0063: case 0x0065: case 0x0067:
                /* PPI */
                ppi_inb(cpssp, valp);
                return 0;

        case 0x0072:
        case 0x0073:
                /* RTC */
                if ((cpssp->rtccfg >> 2) & 1) {
                        rtc_inb(cpssp, valp, port & 3);
                        return 0;
                }
                /*FALLTHROUGH*/
        case 0x0070: case 0x0074: case 0x0076:
        case 0x0071: case 0x0075: case 0x0077:
                rtc_inb(cpssp, valp, port & 1);
                return 0;

        case 0x0080: case 0x0090:
        case 0x0081: case 0x0091:
        case 0x0082: /* 0x0092: special! */
        case 0x0083: case 0x0093:
        case 0x0084: case 0x0094:
        case 0x0085: case 0x0095:
        case 0x0086: case 0x0096:
        case 0x0087: case 0x0097:
        case 0x0088: case 0x0098:
        case 0x0089: case 0x0099:
        case 0x008a: case 0x009a:
        case 0x008b: case 0x009b:
        case 0x008c: case 0x009c:
        case 0x008d: case 0x009d:
        case 0x008e: case 0x009e:
        case 0x008f: case 0x009f:
                *valp = cpssp->dma_page[port & 0xf];
                return 0;

        case 0x00a0: case 0x00a1: case 0x00a4: case 0x00a5:
        case 0x00a8: case 0x00a9: case 0x00ac: case 0x00ad:
        case 0x00b0: case 0x00b1: case 0x00b4: case 0x00b5:
        case 0x00b8: case 0x00b9: case 0x00bc: case 0x00bd:
                pic1_inb(cpssp, valp, port & 0x1);
                return 0;

        case 0x00c0 ... 0x00df:
                dma1_inb(cpssp, valp, (port >> 1) & 0xf);
                return 0;

        case 0x04d0:
                /* PIC0 ELCR - Edge/Level Control Register */
                pic0_inb(cpssp, valp, 2);
                return 0;
        case 0x04d1:
                /* PIC1 ELCR - Edge/Level Control Register */
                pic1_inb(cpssp, valp, 2);
                return 0;

        default:
                break;
        }

        return 1;
}

static int
legacy_outb(struct cpssp *cpssp, uint32_t port, uint8_t val)
{
        switch (port) {
        case 0x0000 ... 0x001f:
                dma0_outb(cpssp, val, port & 0xf);
                return 0;

	case 0x0020: case 0x0021: case 0x0024: case 0x0025:
        case 0x0028: case 0x0029: case 0x002c: case 0x002d:
        case 0x0030: case 0x0031: case 0x0034: case 0x0035:
        case 0x0038: case 0x0039: case 0x003c: case 0x003d:
                pic0_outb(cpssp, val, port & 0x1);
                return 0;

        case 0x0040 ... 0x0043:
        case 0x0050 ... 0x0053:
                /* PIT 0 */
                pit_outb(cpssp, val, port & 0x3);
                return 0;

	case 0x0061: case 0x0063: case 0x0065: case 0x0067:
                /* PPI */
                ppi_outb(cpssp, val);
                return 0;

        case 0x0072:
        case 0x0073:
                /* RTC */
                if ((cpssp->rtccfg >> 2) & 1) {
                        rtc_outb(cpssp, val, port & 3);
                        return 0;
                }
                /*FALLTHROUGH*/
        case 0x0070: case 0x0074: case 0x0076:
        case 0x0071: case 0x0075: case 0x0077:
                rtc_outb(cpssp, val, port & 1);
                return 0;

        case 0x0080: case 0x0090:
        case 0x0081: case 0x0091:
        case 0x0082: /* 0x0092: See below! */
        case 0x0083: case 0x0093:
        case 0x0084: case 0x0094:
        case 0x0085: case 0x0095:
        case 0x0086: case 0x0096:
        case 0x0087: case 0x0097:
        case 0x0088: case 0x0098:
        case 0x0089: case 0x0099:
        case 0x008a: case 0x009a:
        case 0x008b: case 0x009b:
        case 0x008c: case 0x009c:
        case 0x008d: case 0x009d:
        case 0x008e: case 0x009e:
        case 0x008f: case 0x009f:
                cpssp->dma_page[port & 0xf] = val;
                return 0;

	case 0x00a0: case 0x00a1: case 0x00a4: case 0x00a5:
        case 0x00a8: case 0x00a9: case 0x00ac: case 0x00ad:
        case 0x00b0: case 0x00b1: case 0x00b4: case 0x00b5:
        case 0x00b8: case 0x00b9: case 0x00bc: case 0x00bd:
                pic1_outb(cpssp, val, port & 0x1);
                return 0;

       case 0x00c0 ... 0x00df:
                dma1_outb(cpssp, val, (port >> 1) & 0xf);
                return 0;

        case 0x04d0:
                /* PIC0 ELCR - Edge/Level Control Register */
                pic0_outb(cpssp, val, 2);
                return 0;
        case 0x04d1:
                /* PIC1 ELCR - Edge/Level Control Register */
                pic1_outb(cpssp, val, 2);
                return 0;

        default:
                break;
        }

        return 1;
}

static int
legacy_ior(void *_cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t val8;

	switch (bs) {
	case 0b1 << 0:
		if (legacy_inb(cpssp, port + 0, &val8) == 0) {
			*valp &= ~(0xff << 0);
			*valp |= val8 << 0;
			return 0;
		}
		break;
	case 0b1 << 1:
		if (legacy_inb(cpssp, port + 1, &val8) == 0) {
			*valp &= ~(0xff << 8);
			*valp |= val8 << 8;
			return 0;
		}
		break;
	case 0b1 << 2:
		if (legacy_inb(cpssp, port + 2, &val8) == 0) {
			*valp &= ~(0xff << 16);
			*valp |= val8 << 16;
			return 0;
		}
		break;
	case 0b1 << 3:
		if (legacy_inb(cpssp, port + 3, &val8) == 0) {
			*valp &= ~(0xff << 24);
			*valp |= val8 << 24;
			return 0;
		}
		break;
	default:
		break;
	}

	return 1;
}

static int
legacy_iow(void *_cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	switch (bs) {
	case 0b1 << 0:
		return legacy_outb(cpssp, port + 0, (val >> 0) & 0xff);
	case 0b1 << 1:
		return legacy_outb(cpssp, port + 1, (val >> 8) & 0xff);
	case 0b1 << 2:
		return legacy_outb(cpssp, port + 2, (val >> 16) & 0xff);
	case 0b1 << 3:
		return legacy_outb(cpssp, port + 3, (val >> 24) & 0xff);
	}

	return 1;
}

static int
legacy_io_responsible(struct cpssp *cpssp, uint16_t port, unsigned int bs)
{
	switch (bs) {
	case 0b1 << 0: port += 0; break;
	case 0b1 << 1: port += 1; break;
	case 0b1 << 2: port += 2; break;
	case 0b1 << 3: port += 3; break;
	default: return 0;
	}

	switch (port) {
	case 0x0000 ... 0x001f:
		/* DMA 0 */
		return 1;

        case 0x0020: case 0x0021: case 0x0024: case 0x0025:
        case 0x0028: case 0x0029: case 0x002c: case 0x002d:
        case 0x0030: case 0x0031: case 0x0034: case 0x0035:
        case 0x0038: case 0x0039: case 0x003c: case 0x003d:
		/* PIC 0 */
		return 1;

        case 0x0040 ... 0x0043:
        case 0x0050 ... 0x0053:
		/* PIT 0 */
                return 1;

        case 0x0061: case 0x0063: case 0x0065: case 0x0067:
		/* PPI */
		return 1;

	case 0x0070 ... 0x0077:
		/* RTC */
		return 1;

	case 0x0080: case 0x0090:
	case 0x0081: case 0x0091:
	case 0x0082: /* 0x0092: special! */
	case 0x0083: case 0x0093:
	case 0x0084: case 0x0094:
	case 0x0085: case 0x0095:
	case 0x0086: case 0x0096:
	case 0x0087: case 0x0097:
	case 0x0088: case 0x0098:
	case 0x0089: case 0x0099:
	case 0x008a: case 0x009a:
	case 0x008b: case 0x009b:
	case 0x008c: case 0x009c:
	case 0x008d: case 0x009d:
	case 0x008e: case 0x009e:
	case 0x008f: case 0x009f:
		/* DMA Pages */
		return 1;

	case 0x00a0: case 0x00a1: case 0x00a4: case 0x00a5:
	case 0x00a8: case 0x00a9: case 0x00ac: case 0x00ad:
	case 0x00b0: case 0x00b1: case 0x00b4: case 0x00b5:
	case 0x00b8: case 0x00b9: case 0x00bc: case 0x00bd:
		/* PIC 1 */
		return 1;

	case 0x00c0 ... 0x00df:
		/* DMA 1 */
		return 1;

        case 0x04d0:
		/* ELCR 0 */
		return 1;

        case 0x04d1:
		/* ELCR 1 */
		return 1;

	default:
		return 0;
        }
}

static int
legacy_ior_info(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
        void **csp
)
{
	if (legacy_io_responsible(cpssp, addr, bs)) {
		*cfp = legacy_ior;
		*csp = cpssp;
		return 0;
	}
	return 1;
}

static int
legacy_iow_info(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
        void **csp
)
{
	if (legacy_io_responsible(cpssp, addr, bs)) {
		*cfp = legacy_iow;
		*csp = cpssp;
		return 0;
	}
	return 1;
}

static void
legacy_reset(struct cpssp *cpssp)
{
	dma0_reset(cpssp);
	dma1_reset(cpssp);
	pic0_reset(cpssp);
	pic1_reset(cpssp);
	pit_reset(cpssp);
	ppi_reset(cpssp);
        rtc_reset(cpssp);
}

static void
legacy_create(struct cpssp *cpssp, const char *rtc_start, const char *cmos)
{
	dma0_init(cpssp);
        dma1_init(cpssp);
	pic0_create(cpssp);
        pic1_create(cpssp);
	pit_init(cpssp);
	ppi_init(cpssp);
	rtc_create(cpssp, "rtc", rtc_start, cmos);
        rtc_init(cpssp);
}

static void
legacy_destroy(struct cpssp *cpssp)
{
        rtc_destroy(cpssp);
        pic1_destroy(cpssp);
	pic0_destroy(cpssp);
}

#endif /* BEHAVIOR */
