///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// compute the element mass matrix condition number
// for non-regression test purpose
//
// author: Pierre.Saramito@imag.fr
//
// date: 2 september 2017
//
#include "rheolef/skit.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis.h"
#include "form_element.h"
using namespace rheolef;
using namespace std;
#include "arma2csr.icc"

Float one_f (const point& x) { return 1; }

template <class Basis>
void 
show_mass_base (
  const Basis&      b,
  reference_element hat_K,
  arma::Mat<Float>& mass,
  bool              dump)
{
  build_mass (b, hat_K, mass);
  csr<Float,sequential> mass_csr = arma2csr(mass);
  cout << setprecision(numeric_limits<Float>::digits10)
       << "approx   " << b.name() << endl
       << "element  " << hat_K.name() << endl
       << "det      " << det(mass) << endl
       << "cond     " << cond(mass) << endl
       << "size     " << mass.n_rows << endl
       << "nnz      " << mass_csr.dis_nnz()<< endl
       << "fill     " << 1.*mass_csr.dis_nnz()/sqr(mass.n_rows)<< endl
    ;
  if (!dump) return;
  odiststream out ("mass.mm");
  out << mass_csr;
}
void 
show_mass (
  const basis&   b,
  reference_element hat_K,
  bool              dump)
{
  arma::Mat<Float> mass;
  show_mass_base (b, hat_K, mass, dump);
  arma::Col<Float> one_dof;
  b.compute_dof (hat_K, one_f, one_dof);
  arma::Col<Float> meas_hat_K = one_dof.t()*mass*one_dof;
  cout << setprecision(numeric_limits<Float>::digits10)
       << "node     " << b.option().get_node_name() << endl
       << "raw_poly " << b.option().get_raw_polynomial_name() << endl
       << "meas_K   " << meas_hat_K(0,0) << endl
    ;
}
int main(int argc, char**argv) {
  environment rheolef (argc, argv);
  string approx = (argc > 1) ? argv[1]    : "P3";
  char   t      = (argc > 2) ? argv[2][0] : 't';
  bool   dump   = (argc > 3);
  reference_element hat_K;
  hat_K.set_name(t);
  if (approx[0] == 'D') {
    basis_raw b (approx);
    arma::Mat<Float> mass;
    show_mass_base (b, hat_K, mass, dump);
  } else {
    basis b (approx);
    show_mass (b, hat_K, dump);
 }
}
