/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 *
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Similar to
 * u-boot (Version 2009-11),
 *	drivers/spi/davinci_spi.c
 *	drivers/spi/davinci_spi.h
 */

#include "assert.h"

#include "spi.h"

struct spi_regs {
	/* 0x00 */
	unsigned int gcr0;
#define SPIGCR0_SPIENA_MASK	0x1
#define SPIGCR0_SPIRST_MASK	0x0

	unsigned int gcr1;
#define SPIGCR1_SPIENA_MASK	(1 << 24)
#define SPIGCR1_CLKMOD_MASK	(1 << 1)
#define SPIGCR1_MASTER_MASK	(1 << 0)

	unsigned int int0;
	unsigned int lvl;

	/* 0x10 */
	unsigned int flg;
	unsigned int pc0;
#define SPIPC0_DIFUN_MASK	(1 << 11)
#define SPIPC0_DOFUN_MASK	(1 << 10)
#define SPIPC0_CLKFUN_MASK	(1 << 9)
#define SPIPC0_EN0FUN_MASK	(1 << 0)

	unsigned int pc1;
	unsigned int pc2;
	/* 0x20 */
	unsigned int pc3;
	unsigned int pc4;
	unsigned int pc5;

	unsigned int _2c;
	/* 0x30 */
	unsigned int _30;
	unsigned int _34;
	unsigned int dat0;
	unsigned int dat1;
#define SPIDAT1_CSHOLD_SHIFT	28
#define SPIDAT1_CSNR_SHIFT	16

	/* 0x40 */
	unsigned int buf;
#define SPIBUF_RXEMPTY_MASK	(1 << 31)
#define SPIBUF_TXFULL_MASK	(1 << 29)

	unsigned int emu;
	unsigned int delay;
#define SPI_C2TDELAY_SHIFT	24
#define SPI_T2CDELAY_SHIFT	16

	unsigned int def;
#define SPIDEF_CSDEF0_MASK	(1 << 0)

	/* 0x50 */
	unsigned int fmt0;
	unsigned int fmt1;
	unsigned int fmt2;
	unsigned int fmt3;
#define SPIFMT_SHIFTDIR_SHIFT	20
#define SPIFMT_POLARITY_SHIFT	17
#define SPIFMT_PHASE_SHIFT	16
#define SPIFMT_PRESCALE_SHIFT	8

	/* 0x60 */
	unsigned int _60;
	unsigned int ivec;

};
#define REGS ((volatile struct spi_regs *) 0x01c41000)

int
spi_claim_bus(void)
{
	/*
	 * Enable SPI hardware.
	 */
	REGS->gcr0 = SPIGCR0_SPIRST_MASK;
	/* Delay... - FIXME */
	REGS->gcr0 = SPIGCR0_SPIENA_MASK;

	/*
	 * Set master mode, powered up and not activated.
	 */
	REGS->gcr1 = SPIGCR1_MASTER_MASK | SPIGCR1_CLKMOD_MASK;

	/*
	 * CS, CLK, SIMO and SOMI are functional pins.
	 */
	REGS->pc0 = SPIPC0_EN0FUN_MASK | SPIPC0_CLKFUN_MASK
			| SPIPC0_DOFUN_MASK | SPIPC0_DIFUN_MASK;

	/*
	 * Setup format.
	 * Use following format:
	 *   character length = 8,
	 *   clock signal delayed by half clk cycle,
	 *   clock low in idle state - Mode 0,
	 *   MSB shifted out first
	 */
	REGS->fmt0 = 8
			| (1 << SPIFMT_PRESCALE_SHIFT)
			| (1 << SPIFMT_PHASE_SHIFT);

	/*
	 * Hold cs active at end of transfer until explicitly de-asserted.
	 */
	REGS->dat1 = (1 << SPIDAT1_CSHOLD_SHIFT)
			| (0 << SPIDAT1_CSNR_SHIFT);

	/*
	 * Including a minor delay. No science here. Should be good even
	 * with no delay.
	 */
	REGS->delay = (50 << SPI_C2TDELAY_SHIFT)
			| (50 << SPI_T2CDELAY_SHIFT);

	/*
	 * Default chip select register.
	 */
	REGS->def = SPIDEF_CSDEF0_MASK;

	/*
	 * No interrupts.
	 */
	REGS->int0 = 0x00;
	REGS->lvl = 0x00;
	
	/*
	 * Enable SPI.
	 */
	REGS->gcr1 |= SPIGCR1_SPIENA_MASK;

	return 0;
}

void
spi_release_bus(void)
{
	REGS->gcr0 = SPIGCR0_SPIRST_MASK;
}

int
spi_xfer(
	unsigned int bitlen,
	const void *_dout,
	void *_din,
	unsigned long flags
)
{
	const unsigned char *dout = _dout;
	unsigned char *din = _din;
	unsigned int rx_len, tx_len;
	unsigned int status;
	unsigned int data;

	assert(bitlen % 8 == 0);

	rx_len = tx_len = bitlen / 8;

	/*
	 * Do an empty read to clear the current contents.
	 */
	(void) REGS->buf;

	/*
	 * Setup for 8-bit transfer.
	 */
	REGS->fmt0 &= 0xffffffe0;
	REGS->fmt0 |= 8;

	while (0 < rx_len) {
		status = REGS->buf;

		if (! (status & SPIBUF_TXFULL_MASK)
		 && 0 < tx_len) {
			/* CS */
			data = 0 << SPIDAT1_CSNR_SHIFT;

			/* CSHOLD */
			if (tx_len == 1 && (flags & SPI_XFER_END)) {
				/* Last transfer. */
				data &= ~(1 << SPIDAT1_CSHOLD_SHIFT);
			} else {
				/* Not the last transfer. */
				data |= 1 << SPIDAT1_CSHOLD_SHIFT;
			}

			/* DATA */
			if (dout) {
				data |= *dout++;
			}

			REGS->dat1 = data;
			tx_len--;
		}
		if (! (status & SPIBUF_RXEMPTY_MASK)) {
			if (din) {
				*din++ = status & 0xff;
			}
			rx_len--;
		}
	}

	return 0;
}
