/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "voltmeter.h"

#define COMP_(x)	voltmeter_ ## x

#define FREQ	128

struct cpssp {
	unsigned int state_gnd;
	unsigned int state_vdd;
	struct sig_integer *port_voltage;

	int voltage;
	int count;

	unsigned long long time;
};

static void
COMP_(gnd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_gnd = SIG_mV(val);
}

static void
COMP_(vdd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_vdd = SIG_mV(val);
}

static void
COMP_(event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->voltage += cpssp->state_vdd /* - cpssp->state_gnd FIXME */;
	cpssp->count++;

	if (cpssp->count == FREQ) {
		sig_integer_set(cpssp->port_voltage, cpssp,
				cpssp->voltage / cpssp->count);

		cpssp->voltage = 0;
		cpssp->count = 0;
	}

	cpssp->time += TIME_HZ / FREQ;
	time_call_at(cpssp->time, COMP_(event), cpssp);
}

void *
COMP_(create)(
	const char *name,
        struct sig_manage *port_manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vdd,
	struct sig_integer *port_voltage
)
{
	static const struct sig_std_logic_funcs gnd_funcs = {
		.std_logic_set = COMP_(gnd_set),
	};
	static const struct sig_std_logic_funcs vdd_funcs = {
		.std_logic_set = COMP_(vdd_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->voltage = 0;
	cpssp->count = 0;

	/* Out */
	cpssp->port_voltage = port_voltage;

	/* In */
	cpssp->state_gnd = 0;
	sig_std_logic_connect_in(port_gnd, cpssp, &gnd_funcs);
	cpssp->state_vdd = 0;
	sig_std_logic_connect_in(port_vdd, cpssp, &vdd_funcs);

	/* Time */
	cpssp->time = 0;
	time_call_at(cpssp->time, COMP_(event), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
